const { app, BrowserWindow, ipcMain, shell, dialog, screen } = require('electron'); // Ajout de dialog pour les fenêtres de confirmation
const pdfParse = require('pdf-parse'); // Pour lire les fichiers PDF
const archiver = require('archiver'); // Pour la compression
const path = require('path'); // Pour gérer les chemins de fichiers
const fs = require('fs'); // Pour lire les fichiers

/////////
// début de la fonction createWindow
let mainWindow;
let logWindow; // Fenêtre de logs

function createWindow() {
    // Obtenir les informations sur l'écran principal pour que l'application s'ouvre sur le bon écran
    const primaryDisplay = screen.getPrimaryDisplay();
    const { x, y } = primaryDisplay.workArea;

    mainWindow = new BrowserWindow({
        x: x,
        y: y,
        width: 1200,
        height: 730,
        webPreferences: {
            preload: path.join(__dirname, 'preload.js'),
            contextIsolation: true,
            enableRemoteModule: false,
            }
    });

    mainWindow.loadFile('index.html');

    // Masquer le menu principal
    mainWindow.setMenu(null);
    mainWindow.center(); // pour ouvrir la fenêtre de l'application au centre
    //mainWindow.webContents.openDevTools();

    ipcMain.handle('get-translations', async () => {
        const defaultLanguage = 'fr';
        const translations = {
            fr: {
                log_title: "Indexation",
                log_warning: "Attention, cette opération peut prendre du temps suivant le nombre de fichiers que vous avez à traiter dans votre dossier ./files",
                close_button: "Fermer cette fenêtre"
            },
            en: {
                log_title: "Indexing",
                log_warning: "Warning: this operation may take some time depending on the number of files in your ./files directory",
                close_button: "Close this window"
            }
        };
    
        try {
            const selectedLanguage = await mainWindow.webContents.executeJavaScript(
                'localStorage.getItem("selectedLanguage") || "fr"'
            );
            return translations[selectedLanguage] || translations[defaultLanguage];
        } catch (error) {
            console.error('Erreur lors de la récupération de la langue :', error);
            return translations[defaultLanguage];
        }
    });

    mainWindow.webContents.on('did-finish-load', async () => {
        console.log('Fenêtre principale chargée.');
    
        // Charger les traductions selon la langue sélectionnée
        const translations = {
            fr: {
                title: "Voulez-vous lancer une indexation avant de démarrer HomeGED ?",
                detail: "Lancer une indexation peut prendre un certain temps. Vous pouvez choisir de l'ignorer pour démarrer plus rapidement.\nNotez qu'il est toujours possible d'indexer manuellement via le bouton de rafraîchissement (🔄), si besoin.",
                yes: "✔️ Oui",
                no: "❌ Non"
            },
            en: {
                title: "Would you like to start indexing before launching HomeGED?",
                detail: "Starting indexing may take some time. You can skip it to start faster.\nYou can always manually index files later using the refresh button (🔄), if needed.",
                yes: "✔️ Yes",
                no: "❌ No"
            }
        };
    
        // Déterminez la langue actuelle (par défaut : français)
        const currentLanguage = await mainWindow.webContents.executeJavaScript('localStorage.getItem("selectedLanguage") || "fr"');
        const lang = translations[currentLanguage] || translations.fr;
    
        // Afficher la boîte de dialogue avec les traductions chargées
        const response = await dialog.showMessageBox(mainWindow, {
            type: 'question',
            buttons: [lang.yes, lang.no],
            defaultId: 1,
            cancelId: 2, // Gérer la fermeture avec la croix
            message: lang.title,
            detail: lang.detail
        });
    
        if (response.response === 0) { // L'utilisateur a choisi "Oui"
            console.log('L\'Utilisateur a choisi de lancer l\'indexation.');
            createLogWindow(); // Créer la fenêtre de logs avant de commencer l'indexation
            initializeIndexing();
        } else if (response.response === 1) { // L'utilisateur a choisi "Non"
            console.log('L\'Utilisateur a choisi de ne pas lancer l\'indexation.');
        } else if (response.response === 2) { // L'utilisateur a fermé la boîte avec la croix
            console.log('L\'Utilisateur a fermé la boîte de dialogue.');
            // Aucune action
        }
    });
    
    
    ipcMain.on('zoom-in', (event) => {
        console.log('Zoom In command received'); // Vérifier si l'événement est reçu
        if (mainWindow) {
            mainWindow.webContents.send('zoom-in');
        }
    });
    
    ipcMain.on('zoom-out', (event) => {
        console.log('Zoom Out command received'); // Vérifier si l'événement est reçu
        if (mainWindow) {
            mainWindow.webContents.send('zoom-out');
        }
    });
    
    // permet d'ouvrir les url cliquées sans ouvrir une nouvelle fenêtre electron, directement dans le navigateur par défaut
    if (mainWindow && mainWindow.webContents) {
        mainWindow.webContents.setWindowOpenHandler(({ url }) => {
            const { shell } = require('electron');
            if (url.startsWith('http')) {
                shell.openExternal(url);
                return { action: 'deny' };
            }
            return { action: 'allow' };
        });
    } else {
        console.error("mainWindow ou webContents n'est pas défini.");
    }
    
        
}

function createLogWindow() {
    // Obtenir l'écran principal
    const primaryDisplay = screen.getPrimaryDisplay();
    const { x, y } = primaryDisplay.workArea;

    logWindow = new BrowserWindow({
        x: x,
        y: y,
        width: 500,
        height: 460,
        webPreferences: {
            preload: path.join(__dirname, 'preload.js'),
            contextIsolation: true,
            enableRemoteModule: false,
        }
    });

    logWindow.loadFile('log.html');

    logWindow.webContents.on('did-finish-load', () => {
        console.log('Fenêtre de log chargée.');
    });

    logWindow.setMenu(null); // pas de menus électron
    logWindow.center(); // Pour centrer la fenêtre à l'écran
}

// Quand toutes les fenêtres sont fermées, nettoyage de cache et fermeture de l'application
app.on('window-all-closed', async () => {
    if (process.platform !== 'darwin') {
        try {
            await session.defaultSession.clearStorageData({
                storages: ['cookies', 'localstorage', 'indexdb', 'filesystem', 'cachestorage'],
                quotas: ['temporary', 'persistent', 'syncable'],
            });
            console.log('Cache et historique de recherche supprimés avec succès.');
        } catch (error) {
            console.error('Erreur lors de la suppression du cache et des données de session:', error);
        }
        app.quit();
    }
});

app.whenReady().then(() => {
    createWindow();
    
});

app.on('window-all-closed', () => {
    if (process.platform !== 'darwin') {
        app.quit();
    }
});

async function initializeIndexing() {
    try {
        const currentLanguage = await mainWindow.webContents.executeJavaScript('localStorage.getItem("selectedLanguage") || "fr"');
        const translations = {
            fr: {
                preparing: "1 - Préparation de l'indexation...",
                directoryNotExist: "Le répertoire {directory} n'existe pas.",
                startIndexing: "2 - Début de l'indexation des fichiers",
                indexGenerated: "&emsp;- Index actuel généré.",
                indexSaved: "&emsp;- Index sauvegardé avec succès.",
                totalFiles: "Total des fichiers après indexation : {totalFiles}",
                indexingComplete: "<br>(i) Indexation terminée.<br>Vous pouvez maintenant fermer cette fenêtre.",
                indexingError: "<br>(!) Erreur lors de l'initialisation de l'indexation : {error}"
            },
            en: {
                preparing: "1 - Preparing for indexing...",
                directoryNotExist: "The directory {directory} does not exist.",
                startIndexing: "2 - Starting file indexing",
                indexGenerated: "&emsp;- Current index generated.",
                indexSaved: "&emsp;- Index saved successfully.",
                totalFiles: "Total files after indexing: {totalFiles}",
                indexingComplete: "<br>(i) Indexing complete.<br>You can now close this window.",
                indexingError: "<br>(!) Error during indexing initialization: {error}"
            }
        };

        const lang = translations[currentLanguage] || translations.fr;

        if (!logWindow || logWindow.isDestroyed()) {
            createLogWindow();
        }

        await new Promise(resolve => {
            logWindow.webContents.once('did-finish-load', resolve);
        });

        sendLogMessage(lang.preparing);

        if (mainWindow) {
            mainWindow.webContents.send('loading-update', 'start');
        }

        await new Promise(resolve => setTimeout(resolve, 5000));

        const filesDirectory = path.join(__dirname, 'files');
        console.log(`Chemin de filesDirectory : ${filesDirectory}`);

        if (!fs.existsSync(filesDirectory)) {
            const errorMessage = lang.directoryNotExist.replace("{directory}", filesDirectory);
            console.error(errorMessage);
            sendLogMessage(errorMessage);
            return;
        }

        sendLogMessage(lang.startIndexing);

        const currentIndex = await generateFileIndex(filesDirectory);
        sendLogMessage(lang.indexGenerated);

        // Écraser l'index précédent avec le nouvel index
        saveFileIndex(currentIndex);
        sendLogMessage(lang.indexSaved);

        const totalFiles = countFilesInDirectory(filesDirectory);
        console.log(lang.totalFiles.replace("{totalFiles}", totalFiles));
        sendLogMessage(lang.totalFiles.replace("{totalFiles}", totalFiles));

        if (mainWindow) {
            mainWindow.webContents.send('update-file-counter', totalFiles);
        }

        await new Promise(resolve => setTimeout(resolve, 500));

        if (mainWindow) {
            console.log('Envoi du signal refresh-arborescence au Renderer Process.');
            mainWindow.webContents.send('refresh-arborescence');
        }

        sendLogMessage(lang.indexingComplete);
    } catch (error) {
        sendLogMessage(lang.indexingError.replace("{error}", error.message));
    } finally {
        if (mainWindow) {
            mainWindow.webContents.send('loading-update', 'end');
        }

        if (logWindow && !logWindow.isDestroyed()) {
            console.log('Envoi du signal show-close-button à logWindow');
            logWindow.webContents.send('show-close-button');
        } else {
            console.error('logWindow est invalide ou détruite.');
        }
    }
}


ipcMain.handle('close-log-window', () => {
    if (logWindow && !logWindow.isDestroyed()) {
        logWindow.close();
        console.log('Fenêtre de logs fermée. Envoi du signal pour rafraîchir l’arborescence.');
        mainWindow.webContents.send('refresh-arborescence'); // Envoie un signal au Renderer Process
    }
});

// Fonction pour envoyer des logs à la fenêtre de logs
function sendLogMessage(message) {
    console.log(message); // Affiche également dans la console
    if (logWindow && logWindow.webContents) {
        logWindow.webContents.send('log-message', message);
    }
}

// Gestion de la fermeture de la fenêtre de logs par un événement explicite
ipcMain.on('close-log-window', async () => {
    if (logWindow && !logWindow.isDestroyed()) {
        logWindow.close();
        console.log('Fenêtre de logs fermée par un événement explicite.');

        // Rafraîchir l'arborescence après la fermeture de la fenêtre de logs
        if (mainWindow) {
            try {
                console.log('Envoi du signal pour rafraîchir l’arborescence.');
                
                // Scannez les fichiers pour obtenir les données à jour
                const dirPath = path.join(app.getAppPath(), 'files');
                const updatedFiles = scanDirectory(dirPath); // Assurez-vous que scanDirectory fonctionne correctement
                
                // Envoyez les données actualisées au Renderer Process
                mainWindow.webContents.send('refresh-arborescence', updatedFiles);
                console.log('Signal refresh-arborescence envoyé avec succès.');
            } catch (error) {
                console.error('Erreur lors du rafraîchissement de l’arborescence :', error);
            }
        }
    }
});

// Gestion du rafraîchissement de l'index
ipcMain.handle('refresh-index', async () => {
    try {
        console.log("- Début du rafraîchissement de l'index...");
        sendLogMessage('&emsp;- Début du rafraîchissement de l\'index...');
        
        // Créer la fenêtre de logs si elle n'est pas déjà ouverte
        if (!logWindow || logWindow.isDestroyed()) {
            createLogWindow();
        }

        const updatedIndex = await initializeIndexing(); // Rafraîchir l'index
        const filePath = path.join(__dirname, 'file_index.json');
        fs.writeFileSync(filePath, JSON.stringify(updatedIndex, null, 4), 'utf-8');
        
        console.log("Index mis à jour avec succès.");
        sendLogMessage('<br>(i) Index mis à jour avec succès.');
        
        return updatedIndex;
    } catch (error) {
        console.error("Erreur lors du rafraîchissement de l'index :", error);
        sendLogMessage(`<br>(!) Erreur lors du rafraîchissement de l'index : ${error.message}`);
        throw error;
    }
});

// Gestion de la fermeture de toutes les fenêtres
app.on('window-all-closed', () => {
    if (process.platform !== 'darwin') {
        app.quit();
    }
});


ipcMain.handle('show-refresh-dialog', async () => {
    // Charger les traductions en fonction de la langue actuelle
    const translations = {
        fr: {
            title: "Souhaitez-vous effectuer un rafraîchissement avec réindexation ou sans réindexation ?",
            detail: "Le rafraîchissement avec réindexation met à jour l'index des fichiers mais prend plus de temps.",
            withReindexing: "✔️ Rafraîchir avec réindexation",
            withoutReindexing: "❌ Rafraîchir sans réindexation",
            cancel: "Annuler"
        },
        en: {
            title: "Would you like to refresh with reindexing or without reindexing?",
            detail: "Refreshing with reindexing updates the file index but takes more time.",
            withReindexing: "✔️ Refresh with reindexing",
            withoutReindexing: "❌ Refresh without reindexing",
            cancel: "Cancel"
        }
    };

    // Détermine la langue actuelle (par défaut : français)
    const currentLanguage = await mainWindow.webContents.executeJavaScript(
        'localStorage.getItem("selectedLanguage") || "fr"'
    );
    const lang = translations[currentLanguage] || translations.fr;

    // Afficher la boîte de dialogue avec les traductions appropriées
    const response = await dialog.showMessageBox(mainWindow, {
        type: 'question',
        buttons: [lang.withReindexing, lang.withoutReindexing, lang.cancel],
        defaultId: 2, // Le bouton "Annuler" est le bouton par défaut
        cancelId: 2, // La croix de fermeture est équivalente à "Annuler"
        message: lang.title,
        detail: lang.detail
    });

    if (response.response === 0) {
        console.log("L'utilisateur a choisi : Rafraîchir avec réindexation.");
        return 0; // Rafraîchir avec réindexation
    } else if (response.response === 1) {
        console.log("L'utilisateur a choisi : Rafraîchir sans réindexation.");
        return 1; // Rafraîchir sans réindexation
    } else {
        console.log("L'utilisateur a annulé l'action.");
        return 2; // Annuler
    }
});



function loadPreviousIndex() {
    const indexFilePath = path.join(__dirname, 'file_index.json');
    console.log('Chemin absolu du fichier d\'index :', indexFilePath);
    sendLogMessage(`&emsp;- Chemin absolu du fichier d'index : ${indexFilePath}`);
    try {
        if (fs.existsSync(indexFilePath)) {
            const indexData = JSON.parse(fs.readFileSync(indexFilePath, 'utf-8'));
            console.log("Données de l'index chargées depuis le fichier :", indexData);
            sendLogMessage("&emsp;- Données de l'index chargées depuis les fichiers dans ./files");
            return indexData;
        } else {
            console.log("Aucun index précédent trouvé.");
            sendLogMessage("&emsp;(i) Aucun index précédent trouvé.");
            return {}; // Retourne un objet vide si le fichier n'existe pas
        }
    } catch (error) {
        console.error("Erreur lors du chargement de l'index précédent :", error);
        sendLogMessage("(!) Erreur lors du chargement de l'index précédent :", error);
        return {}; // En cas d'erreur, retourne un objet vide
    }
}

/////////

// Définit le répertoire fixe des fichiers GED
const filesDirectory = path.join(__dirname, 'files');

// Chemin vers le fichier d'index
const indexFilePath = path.join(__dirname, 'file_index.json');
console.log("Chemin du fichier d'index :", indexFilePath);
sendLogMessage("&emsp;- Chemin du fichier d'index :", indexFilePath);

const crypto = require('crypto'); // Pour générer des hachages

function generateFileHash(filePath) {
    const fileBuffer = fs.readFileSync(filePath);
    const hashSum = crypto.createHash('sha256');
    hashSum.update(fileBuffer);
    return hashSum.digest('hex');
}

function getModifiedFiles(directory, previousIndex = {}) {
    const modifiedFiles = [];
    const currentIndex = {};

    const files = fs.readdirSync(directory, { withFileTypes: true });
    files.forEach(file => {
        const fullPath = path.join(directory, file.name);

        if (file.isFile()) {
            const stats = fs.statSync(fullPath);
            const mtime = stats.mtime.getTime();
            const hash = generateFileHash(fullPath);

            let content = '';

            // Vérifie si le fichier est nouveau ou modifié
            const prevEntry = previousIndex[fullPath];
            if (!prevEntry || prevEntry.mtime !== mtime || prevEntry.hash !== hash) {
                console.log(`Fichier ajouté/modifié : ${fullPath}`);

                try {
                    const extension = path.extname(file.name).toLowerCase();
                    if (extension === '.pdf') {
                        const dataBuffer = fs.readFileSync(fullPath);
                        const pdfData = pdfParse(dataBuffer);
                        content = pdfData.text.toLowerCase();
                    } else if (fullPath.match(/\.(txt|md|html|js|json|docx)$/)) {
                        content = fs.readFileSync(fullPath, 'utf-8').toLowerCase();
                    }
                } catch (error) {
                    console.error(`Erreur lors de la lecture du fichier ${fullPath} :`, error);
                }

                modifiedFiles.push({ file: fullPath, mtime, hash, content });
            } else {
                console.log(`Fichier inchangé, récupération de l'index précédent : ${fullPath}`);
                sendLogMessage(`Fichier inchangé, récupération de l'index précédent : ${fullPath}`);
            }

            // Ajoute le fichier à l'index actuel (sans tags)
            currentIndex[fullPath] = { mtime, hash, content };
        } else if (file.isDirectory()) {
            const subDirResult = getModifiedFiles(fullPath, previousIndex);
            modifiedFiles.push(...subDirResult.modifiedFiles);
            Object.assign(currentIndex, subDirResult.currentIndex);
        }
    });

    return { modifiedFiles, currentIndex };
}

function saveIndex(index) {
    fs.writeFileSync(indexFilePath, JSON.stringify(index, null, 4), 'utf-8');
}

// Afficher les fichiers modifiés
// Suppression du traitement des fichiers modifiés lors du démarrage
async function generateFileIndex(directory, previousIndex = {}) {
    const indexData = {};
    console.log(`Indexation des fichiers dans : ${directory}`);
    sendLogMessage(`&emsp;- Indexation des fichiers dans : ${directory}`);

    const files = fs.readdirSync(directory, { withFileTypes: true });

    for (const file of files) {
        const fullPath = path.join(directory, file.name);
        const stats = fs.statSync(fullPath);

        if (file.isDirectory()) {
            // Réindexer récursivement les sous-répertoires
            const subdirectoryIndex = await generateFileIndex(fullPath, previousIndex);
            Object.assign(indexData, subdirectoryIndex);
            continue;
        }

        if (file.isFile()) {
            const relativePath = path.relative(filesDirectory, fullPath).replace(/\\/g, '/');
            const mtime = stats.mtime.getTime();
            let content = '';

            // Vérifier si le fichier est déjà indexé et inchangé
            if (previousIndex[relativePath] && previousIndex[relativePath].mtime === mtime) {
                console.log(`Fichier inchangé, conservé dans l'index : ${relativePath}`);
                indexData[relativePath] = previousIndex[relativePath];
                continue; // On conserve l'entrée existante
            }

            // Lire le contenu du fichier en fonction de son type
            try {
                const extension = path.extname(file.name).toLowerCase();
                if (extension === '.pdf') {
                    const dataBuffer = fs.readFileSync(fullPath);
                    const pdfData = await pdfParse(dataBuffer);
                    content = pdfData.text.toLowerCase();
                } else if (fullPath.match(/\.(txt|md|html|js|json|docx)$/)) {
                    content = fs.readFileSync(fullPath, 'utf-8').toLowerCase();
                }
            } catch (error) {
                console.error(`Erreur de lecture du fichier ${fullPath} :`, error);
            }

            // Ajouter ou mettre à jour l'entrée dans l'index (avec un chemin relatif)
            indexData[relativePath] = {
                mtime: mtime,
                content: content,
            };
        }
    }

    // Supprimer les fichiers supprimés de l'index
    Object.keys(previousIndex).forEach(filePath => {
        if (!fs.existsSync(path.join(filesDirectory, filePath))) {
            console.log(`Fichier supprimé, retiré de l'index : ${filePath}`);
            delete indexData[filePath];
        }
    });

    return indexData;
}


// Fonction pour sauvegarder l'index dans un fichier JSON
function saveFileIndex(indexData) {
    try {
        if (typeof indexData !== 'object') {
            throw new Error("Données d'index invalides.");
        }

        // Supprimer les chemins absolus du JSON et dédupliquer
        const filteredIndex = Object.keys(indexData).reduce((acc, key) => {
            // Convertir les chemins absolus en chemins relatifs
            const normalizedKey = path.isAbsolute(key)
                ? path.relative(filesDirectory, key).replace(/\\/g, '/')
                : key;

            // Éviter les doublons, conserver le plus récent
            if (!acc[normalizedKey] || acc[normalizedKey].mtime < indexData[key].mtime) {
                acc[normalizedKey] = indexData[key];
            }

            return acc;
        }, {});

        // Sauvegarder l'index nettoyé et dédupliqué
        fs.writeFileSync(indexFilePath, JSON.stringify(filteredIndex, null, 4), 'utf-8');

    } catch (error) {
        console.error("Erreur lors de la sauvegarde de l'index :", error);
    }
}


// Fonction de recherche
function searchInIndex(searchTerm) {
    const indexFilePath = path.join(__dirname, 'file_index.json');
    try {
        console.log(`Terme de recherche reçu : ${searchTerm}`);

        if (!fs.existsSync(indexFilePath)) {
            console.error("Le fichier file_index.json n'existe pas. Chemin vérifié :", indexFilePath);
            return [];
        }

        // Charger l'index depuis le fichier
        const indexData = JSON.parse(fs.readFileSync(indexFilePath, 'utf-8'));
        const indexArray = Object.keys(indexData).map(key => ({
            file: key,
            ...indexData[key]
        }));

        console.log("Index chargé :", indexArray);

        const searchTermLower = searchTerm.toLowerCase();

        // Recherche dans le contenu, le nom du fichier ou les tags
        const results = indexArray.filter(entry => {
            const fileName = entry.file.toLowerCase();

            // Vérifier la présence de tags avant d'y accéder
            const hasMatchingTags = entry.tags && Array.isArray(entry.tags) && entry.tags.some(tag => tag.toLowerCase().includes(searchTermLower));

            return (
                fileName.includes(searchTermLower) ||
                entry.content.includes(searchTermLower) ||
                hasMatchingTags
            );
        });

        console.log("Résultats trouvés avant correction :", results);

        // Correction : Remplacer les fichiers `.json` par leurs fichiers principaux
        const correctedResults = results.map(result => {
            if (result.file.endsWith('.json')) {
                const mainFileName = result.file.replace('.json', '');
                const mainFile = indexArray.find(entry => entry.file === mainFileName);
                return mainFile || result; // Retourne le fichier principal s'il existe, sinon le fichier `.json`
            }
            return result; // Retourne tel quel si ce n'est pas un `.json`
        });

        console.log("Résultats corrigés :", correctedResults);
        return correctedResults;
    } catch (error) {
        console.error("Erreur lors de la recherche :", error);
        return [];
    }
}

// Handler IPC pour 'search-files'
ipcMain.handle('search-files', async (event, searchTerm) => {
    console.log("Terme de recherche reçu :", searchTerm);
    try {
        const results = searchInIndex(searchTerm);
        if (!Array.isArray(results)) {
            throw new Error("Les résultats de la recherche ne sont pas sous forme de tableau.");
        }
        console.log("Résultats envoyés au frontend :", results);
        return results;
    } catch (error) {
        console.error("Erreur lors de la recherche :", error);
        return [];
    }
});

// Enregistre le gestionnaire pour initialize-indexing
ipcMain.handle('initialize-indexing', async () => {
    try {
        console.log("Début de l'indexation...");
        await initializeIndexing();
        const updatedFiles = scanDirectory(path.join(app.getAppPath(), 'files'));
        mainWindow.webContents.send('refresh-arborescence', updatedFiles);
        console.log("Indexation terminée et arborescence mise à jour.");
    } catch (error) {
        console.error("Erreur lors de l'indexation :", error);
    }
});






app.on('window-all-closed', () => {
    if (process.platform !== 'darwin') {
        app.quit();
    }
});

ipcMain.handle('get-files', async (event) => {
    const dirPath = path.join(app.getAppPath(), 'files');
    return scanDirectory(dirPath);
});

ipcMain.handle('open-containing-folder', async (event, filePath) => {
    try {
        const fullPath = path.resolve(__dirname, filePath);
        shell.showItemInFolder(fullPath);
    } catch (error) {
        console.error(`Erreur lors de l'ouverture du dossier contenant le fichier : ${filePath}`, error);
    }
});

//compteur de fichiers
ipcMain.handle('count-files', async (event) => {
    const dirPath = path.join(__dirname, 'files');
    
    function scanDirectory(dir) {
        let fileCount = 0;
    
        // Vérifiez si le répertoire existe
        if (!fs.existsSync(dir)) {
            console.warn(`Le répertoire ${dir} n'existe pas.`);
            return 0; // Retourne 0 si le répertoire n'existe pas
        }
    
        // Lire les fichiers et dossiers dans le répertoire
        const files = fs.readdirSync(dir, { withFileTypes: true });
    
        // Parcourez chaque fichier ou dossier
        files.forEach((file) => {
            const fullPath = path.join(dir, file.name);
    
            if (file.isDirectory()) {
                // Comptage récursif pour les sous-dossiers
                fileCount += scanDirectory(fullPath);
            } else if (!file.name.endsWith('.json')) {
                // Incrémente le compteur pour chaque fichier (excluant les .json)
                fileCount++;
            }
        });
    
        return fileCount; // Retourne le nombre total de fichiers
    }
    
    try {
        const totalFiles = scanDirectory(dirPath);
        console.log(`Total des fichiers comptés : ${totalFiles}`);
        return totalFiles;
    } catch (error) {
        console.error(`Erreur lors du comptage des fichiers : ${error.message}`);
        return 0;
    }
});

// Handler pour lire le contenu d'un fichier
ipcMain.handle('read-file', async (event, filePath, options) => {
    const fs = require('fs').promises;
    const absoluteFilePath = path.join(app.getAppPath(), filePath);

    try {
        if (options.asArrayBuffer) {
            const data = await fs.readFile(absoluteFilePath);
            return data.buffer;
        } else {
            return await fs.readFile(absoluteFilePath, 'utf-8');
        }
    } catch (error) {
        console.error(`Erreur lors de la lecture du fichier : ${absoluteFilePath}`, error);
        throw error;
    }
});

// Handlers pour lire et écrire les fichiers JSON
ipcMain.handle('read-json-file', async (event, jsonFilePath) => {
    try {
        const fullJsonFilePath = path.join(__dirname, jsonFilePath);
        if (fs.existsSync(fullJsonFilePath)) {
            const data = fs.readFileSync(fullJsonFilePath, 'utf-8');
            return JSON.parse(data);
        } else {
            console.log(`Fichier JSON non trouvé: ${fullJsonFilePath}`);
            return {}; // Retourne un objet vide si le fichier n'existe pas
        }
    } catch (error) {
        console.error(`Erreur lors de la lecture du fichier JSON: ${fullJsonFilePath}. Détails: ${error.message}`);
        return null;
    }
});

ipcMain.on('read-json-file-sync', (event, jsonFilePath) => {
    try {
        const fullJsonFilePath = path.join(__dirname, jsonFilePath);
        if (fs.existsSync(fullJsonFilePath)) {
            const data = fs.readFileSync(fullJsonFilePath, 'utf-8');
            event.returnValue = JSON.parse(data);
        } else {
            event.returnValue = {}; // Retourne un objet vide si le fichier n'existe pas
        }
    } catch (error) {
        console.error(`Erreur lors de la lecture du fichier JSON: ${jsonFilePath}. Détails: ${error.message}`);
        event.returnValue = {}; // Retourne un objet vide en cas d'erreur
    }
});

ipcMain.handle('write-json-file', async (event, jsonFilePath, data) => {
    try {
        const fullJsonFilePath = path.join(__dirname, jsonFilePath);
        const dir = path.dirname(fullJsonFilePath);
        if (!fs.existsSync(dir)) {
            fs.mkdirSync(dir, { recursive: true });
        }
        fs.writeFileSync(fullJsonFilePath, JSON.stringify(data, null, 4), 'utf-8');
    } catch (error) {
        console.error(`Erreur lors de l'écriture dans le fichier JSON: ${fullJsonFilePath}. Détails: ${error.message}`);
    }
});

//Fonction pour lire les .html
ipcMain.handle('read-html-file', async (event, filePath) => {
    const absoluteFilePath = path.join(app.getAppPath(), filePath);  // Chemin absolu

    try {
        const content = await fs.promises.readFile(absoluteFilePath, 'utf-8');
        return content;
    } catch (error) {
        console.error(`Erreur lors de la lecture du fichier HTML : ${absoluteFilePath}`, error);
        throw error;
    }
});

// Fonction pour scanner un répertoire
function scanDirectory(dir) {
    const files = fs.readdirSync(dir, { withFileTypes: true });
    const fileStructure = {};

    files.forEach((file) => {
        const fullPath = path.join(dir, file.name);
        if (file.isDirectory()) {
            fileStructure[file.name] = scanDirectory(fullPath);
        } else if (!file.name.endsWith('.json')) {  // Exclure les fichiers JSON
            fileStructure[file.name] = null;
        }
    });

    return fileStructure;
}

function countFilesInDirectory(dir) {
    let fileCount = 0;

    if (!fs.existsSync(dir)) {
        console.warn(`Le répertoire ${dir} n'existe pas.`);
        return 0;
    }

    const files = fs.readdirSync(dir, { withFileTypes: true });

    files.forEach((file) => {
        const fullPath = path.join(dir, file.name);

        if (file.isDirectory()) {
            fileCount += countFilesInDirectory(fullPath);
        } else if (!file.name.endsWith('.json')) { // Exclure les fichiers JSON
            fileCount++;
        }
    });

    return fileCount;
}

// Fonction pour quitter l'application
ipcMain.on('quit-app', () => {
    app.quit();
});

// Configuration de la sécurité du contenu web
app.on('web-contents-created', (event, webContents) => {
    webContents.on('will-attach-webview', (event, webPreferences, params) => {
        delete webPreferences.preload;
        webPreferences.nodeIntegration = false;
    });

    webContents.on('will-navigate', (event, navigationUrl) => {
        const parsedUrl = new URL(navigationUrl);
        if (parsedUrl.origin !== 'file://' && parsedUrl.origin !== 'https://trusted-site.com') {
            event.preventDefault();
        }
    });
});

// Fonction pour créer une sauvegarde du dossier /files
ipcMain.handle('backup-files', async (event) => {
    const filesDir = path.join(app.getAppPath(), 'files');
    const outputFilePath = path.join(app.getAppPath(), 'backup', `files-backup-${Date.now()}.tar.gz`);

    // Crée le répertoire de sauvegarde s'il n'existe pas
    if (!fs.existsSync(path.join(app.getAppPath(), 'backup'))) {
        fs.mkdirSync(path.join(app.getAppPath(), 'backup'));
    }

    return new Promise((resolve, reject) => {
        const output = fs.createWriteStream(outputFilePath);
        const archive = archiver('tar', {
            gzip: true,
            gzipOptions: {
                level: 9
            }
        });

        output.on('close', () => {
            console.log(`Backup completed: ${archive.pointer()} total bytes`);
            // Envoyer un signal de fin au renderer pour signaler la fin de la sauvegarde
            event.sender.send('backup-complete');
            resolve(outputFilePath);
        });

        archive.on('error', (err) => {
            console.error('Error during backup:', err);
            reject(err);
        });

        // Suivi de progression
        archive.on('progress', (data) => {
            const progressPercent = Math.round((data.fs.processedBytes / data.fs.totalBytes) * 100);
            event.sender.send('backup-progress', progressPercent);
        });

        archive.pipe(output);

        // Ajoute les fichiers du répertoire à l'archive
        archive.directory(filesDir, false);

        // Finalise l'archive
        archive.finalize();
    });
});

// capture des évenements de zoom
ipcMain.on('zoom-in', (event) => {
    
    if (mainWindow) {
        mainWindow.webContents.send('zoom-in');
    }
});

ipcMain.on('zoom-out', (event) => {
    
    if (mainWindow) {
        mainWindow.webContents.send('zoom-out');
    }
});

// EOF