// Variables globales pour stocker le fichier actuellement sélectionné et la structure du fichier
let currentFile = null;
let lastContentBuffer = null;
let isDragging = false;
let startX, startY, scrollLeft, scrollTop;

// ajout 1.5.0 - traduction
let currentLanguage = "fr"; // Langue par défaut
let translations = {};

// Fonction pour ouvrir la modale des paramètres
function openSettingsModal() {
    document.getElementById("settingsModal").style.display = "block";
    document.getElementById("modalOverlay").style.display = "block"; // Afficher l'overlay
    updateTranslations(); // Re-appliquer les traductions dans la modale
}

// Fonction pour fermer la modale des paramètres
function closeSettingsModal() {
    document.getElementById("settingsModal").style.display = "none";
    document.getElementById("modalOverlay").style.display = "none"; // Cacher l'overlay
}

// Charger les traductions depuis un fichier JSON
async function loadLanguage(lang) {
    try {
        const response = await fetch(`./langue/${lang}.json`);
        if (!response.ok) {
            throw new Error(`Erreur de chargement : ${response.status}`);
        }
        translations = await response.json(); // Stocker les traductions
        updateTranslations(); // Mettre à jour les textes traduits
        console.log(`Langue ${lang} chargée avec succès.`);
    } catch (err) {
        console.error("Erreur lors du chargement des fichiers de langue :", err);
    }
}

function updateDynamicTexts() {
    const fileCounterElement = document.getElementById('fileCounter');
    if (!fileCounterElement) {
        console.error("Élément fileCounter introuvable !");
        return;
    }

    const fileCount = fileCounterElement.dataset.fileCount; // Récupérer la valeur stockée
    const template = translations["file_counter"]; // Texte traduit
    const currentHTML = fileCounterElement.innerHTML;

    console.log("Valeur actuelle de fileCount :", fileCount);
    console.log("Traduction pour file_counter :", template);

    // Ne modifiez que si le texte contient encore "XX" ou est incorrect
    if (template && fileCount && currentHTML.includes("XX")) {
        fileCounterElement.innerHTML = template.replace("XX", fileCount);
        console.log("updateDynamicTexts : Texte mis à jour :", fileCounterElement.innerHTML);
    } else {
        console.log("updateDynamicTexts : Aucun changement nécessaire.");
    }
}


// Mettre à jour les textes traduisibles dans la page
function updateTranslations() {
    // Mettre à jour les textes statiques
    document.querySelectorAll("[data-translate]").forEach(element => {
        const key = element.getAttribute("data-translate");
        if (translations[key]) {
            element.textContent = translations[key];
        }
    });

    // Mettre à jour les titres (attributs `title`)
    document.querySelectorAll("[data-translate-title]").forEach(element => {
        const key = element.getAttribute("data-translate-title");
        if (translations[key]) {
            element.setAttribute("title", translations[key]);
        } else {
            console.warn(`Clé de traduction pour le titre manquante : ${key}`);
        }
    });

    // Mettre à jour les placeholders
    document.querySelectorAll("[data-translate-placeholder]").forEach(element => {
        const key = element.getAttribute("data-translate-placeholder");
        if (translations[key]) {
            element.setAttribute("placeholder", translations[key]);
        }
    });

    // Mettre à jour les textes dynamiques
    updateDynamicTexts();
}

// Changer la langue et sauvegarder la préférence
function changeLanguage(lang) {
    currentLanguage = lang;
    localStorage.setItem("selectedLanguage", lang); // Sauvegarder la langue dans le localStorage

    loadLanguage(lang).then(() => {
        updateTranslations(); // Mettre à jour les traductions
        updateTitleWithFlag(); // Mettre à jour le drapeau dans le titre
    }).catch(err => {
        console.error("Erreur lors du changement de langue :", err);
    });
}

// Mettre à jour le titre avec le drapeau et le titre traduit
function updateTitleWithFlag() {
    const titleElement = document.querySelector("title");
    const flagEmoji = currentLanguage === "fr" ? "🇫🇷" : "🇺🇸"; // Emoji pour les langues
    const appTitle = translations["app_title"] || "HomeGED v.1.5.0"; // Texte traduit ou défaut

    console.log(`Mise à jour du titre : ${flagEmoji} ${appTitle}`); // Log pour vérifier
    titleElement.textContent = `${flagEmoji} ${appTitle}`;
}


// Initialiser l'application au chargement du DOM
document.addEventListener("DOMContentLoaded", async () => {
    // Restaurer la langue sauvegardée ou utiliser la langue par défaut
    const savedLanguage = localStorage.getItem("selectedLanguage") || "fr"; // Français par défaut
    currentLanguage = savedLanguage;

    // Charger la langue au démarrage
    await loadLanguage(currentLanguage);

    // Mettre à jour les textes traduisibles et le titre
    updateTranslations();
    updateTitleWithFlag();

    // Associer les radios de langue à `changeLanguage`
    document.querySelectorAll('input[name="language"]').forEach(radio => {
        radio.addEventListener('change', (event) => {
            changeLanguage(event.target.value); // Appeler la fonction lors du changement
        });

        // Cocher la bonne radio selon la langue restaurée
        if (radio.value === currentLanguage) {
            radio.checked = true;
        }
    });

    // Ajouter un événement pour l'ouverture de la modale des paramètres
    const settingsIcon = document.querySelector('.settings-icon');
    if (settingsIcon) {
        settingsIcon.addEventListener('click', openSettingsModal);
    }
});


// script global
document.addEventListener('DOMContentLoaded', async function () {
    try {
        // Charge l'arborescence initiale
        let data = await window.electronAPI.getFiles();
        console.log("Données initiales reçues :", data);
        displayDirectoryStructure(data);
        resetTreeView();

        // Attendre que les traductions soient disponibles
        if (!translations) {
            console.warn("Translations non chargées. Retardement...");
            await new Promise(resolve => setTimeout(resolve, 500));
        }

        // Compte les fichiers
        const fileCount = await window.electronAPI.countFiles();

        // Mettre à jour le compteur de fichiers avec traduction
        const fileCounterElement = document.getElementById('fileCounter');
        if (fileCounterElement) {
            const template = translations["file_counter"] || "Nb de fichiers en GED: <b>XX</b>";
            fileCounterElement.dataset.fileCount = fileCount;
            fileCounterElement.innerHTML = template.replace("XX", fileCount);
            console.log(`Compteur mis à jour : ${fileCounterElement.innerHTML}`);
        }

        // Rafraîchir l’arborescence
        console.log('Rafraîchissement de l’arborescence en cours...');
        const refreshedData = await window.electronAPI.getFiles(); // Récupérer les données de l'arborescence
        if (refreshedData && typeof refreshedData === 'object') {
            console.log("Données rafraîchies pour l'arborescence :", refreshedData);
            displayDirectoryStructure(refreshedData); // Met à jour l'affichage
            resetTreeView(); // Réinitialise les états de l'arborescence (repliée ou non)
            console.log('Arborescence rafraîchie avec succès.');
        } else {
            throw new Error("Les données retournées pour l'arborescence sont invalides ou nulles.");
        }
    } catch (error) {
        console.error("Erreur lors de l'initialisation ou du rafraîchissement :", error);

        // Afficher un message d'erreur si nécessaire
        const fileCounterElement = document.getElementById('fileCounter');
        if (fileCounterElement) {
            fileCounterElement.textContent = "Erreur lors du rafraîchissement de l'arborescence.";
        }
    }
});


    // Ajout des événements de zoom après le chargement du DOM
    const zoomInButton = document.getElementById('zoom-in-button');
    const zoomOutButton = document.getElementById('zoom-out-button');

    if (zoomInButton) {
        console.log("Bouton Zoom + trouvé");
        zoomInButton.addEventListener('click', () => {
            console.log("Bouton Zoom + cliqué");
            window.electronAPI.zoomIn(); // Utiliser l'API exposée
        });
    } else {
        console.error("Bouton Zoom + non trouvé");
    }

    if (zoomOutButton) {
        console.log("Bouton Zoom - trouvé");
        zoomOutButton.addEventListener('click', () => {
            console.log("Bouton Zoom - cliqué");
            window.electronAPI.zoomOut(); // Utiliser l'API exposée
        });
    } else {
        console.error("Bouton Zoom - non trouvé");
    }

    // Gestion des événements envoyés par le Main Process pour zoom in et zoom out
    window.electronAPI.receive('zoom-in', () => {
        const previewElement = document.querySelector('#preview canvas, #preview img');
        if (previewElement) {
            let currentZoom = parseFloat(previewElement.getAttribute('data-scale')) || 1;
            currentZoom += 0.1;
            previewElement.style.transform = `scale(${currentZoom})`;
            previewElement.setAttribute('data-scale', currentZoom);
            previewElement.style.transformOrigin = 'top left'; // Assure un zoom fluide
        }
    });
    
    window.electronAPI.receive('zoom-out', () => {
        const previewElement = document.querySelector('#preview canvas, #preview img');
        if (previewElement) {
            let currentZoom = parseFloat(previewElement.getAttribute('data-scale')) || 1;
            currentZoom = Math.max(0.2, currentZoom - 0.1);
            previewElement.style.transform = `scale(${currentZoom})`;
            previewElement.setAttribute('data-scale', currentZoom);
            previewElement.style.transformOrigin = 'top left'; // Assure un zoom fluide
        }
    });

    window.electronAPI.receive('refresh-file-tree', (data) => {
        displayDirectoryStructure(data);
    });

    
   // Écoute les mises à jour du compteur de fichiers
    window.electronAPI.receive('update-file-counter', (fileCount) => {
        const fileCounterElement = document.getElementById('fileCounter');
        if (fileCounterElement) {
            // Utilise le texte traduit
            const template = translations["file_counter"] || "Nb de fichiers en GED: <b>XX</b>";
            fileCounterElement.dataset.fileCount = fileCount;
            fileCounterElement.innerHTML = template.replace("XX", fileCount);
        }
    });

    
// fonction pour mise à jour du compteur en fonction de la langue
async function updateFileCounter() {
    try {
        const fileCounterElement = document.getElementById('fileCounter');
        if (!fileCounterElement) return;

        const fileCount = await window.electronAPI.countFiles();
        const template = translations?.["file_counter"] || "Nb de fichiers en GED: XX";
        fileCounterElement.dataset.fileCount = fileCount; // Stocker la valeur
        fileCounterElement.innerHTML = template.replace("XX", fileCount); // Mettre à jour le HTML
    } catch (error) {
        console.error("Erreur lors de la mise à jour du compteur de fichiers :", error);
        const fileCounterElement = document.getElementById('fileCounter');
        if (fileCounterElement) {
            const errorTemplate = translations?.["file_counter_error"] || "Erreur de chargement des fichiers";
            fileCounterElement.textContent = errorTemplate;
        }
    }
}



// action du clic sur les liens dans les documents
document.getElementById('preview').addEventListener('click', (event) => {
    const target = event.target;

    if (target.tagName === 'A' && target.href) {
        event.preventDefault(); // Empêche l'ouverture dans l'application Electron
        window.electronAPI.openExternal(target.href); // Ouvre dans le navigateur par défaut
    }
});


// compteur de fichiers
function scanDirectory(dir) {
    const fs = require('fs');
    const path = require('path');
    let fileCount = 0;

    try {
        const files = fs.readdirSync(dir, { withFileTypes: true });
        
        files.forEach((file) => {
            const fullPath = path.join(dir, file.name);
            if (file.isDirectory()) {
                fileCount += scanDirectory(fullPath); // Ajoute le nombre de fichiers dans le sous-dossier
            } else if (!file.name.endsWith('.json')) {  // Exclut les fichiers JSON ... pas nécessaire
                fileCount++;
            }
        });
    } catch (error) {
        console.error(`Erreur lors du scan du répertoire : ${dir}`, error);
    }

    return fileCount;
}

function displayDirectoryStructure(data) {
    console.log("Affichage de l'arborescence avec les données :", data);
    const treeDiv = document.getElementById('tree');
    if (!treeDiv) {
        console.error("Élément #tree introuvable dans le DOM !");
        return;
    }
    treeDiv.innerHTML = generateTreeHtml(data);
    console.log("Arborescence mise à jour.");
}


function generateTreeHtml(node, parentPath = '') {
    let html = '<ul>';

    for (let key in node) {
        const itemPath = parentPath ? `${parentPath}/${key}` : key;

        if (node[key] !== null && typeof node[key] === 'object') {
            // Dossier
            html += `<li class="folder">
                        <span class="folder-toggle" onclick="toggleFolder(this, event)">
                            <i class="fas fa-folder"></i>
                            <span class="scrollable-wrapper">
                                <span class="scrollable" 
                                      onmouseover="startScroll(this)" 
                                      onmouseout="stopScroll(this)">
                                      ${key}
                                </span>
                            </span>
                        </span>
                        <div class="folder-contents" style="display: none;">
                            ${generateTreeHtml(node[key], itemPath)}
                        </div>
                    </li>`;
        } else {
            // Fichier
            const iconClass = getFileIconClass(key);
            html += `<li class="file" data-path="${itemPath}">
                        <span class="scrollable-wrapper">
                            <span class="scrollable" 
                                  onmouseover="startScroll(this)" 
                                  onmouseout="stopScroll(this)" 
                                  onclick="openFile('${itemPath}', event)">
                                  <i class="${iconClass}"></i> ${key}
                            </span>
                        </span>
                    </li>`;
        }
    }
    html += '</ul>';
    return html;
}

// Fonction pour démarrer le défilement automatique
function startScroll(element) {
    const wrapperWidth = element.parentElement.offsetWidth; // Largeur visible du conteneur parent
    const textWidth = element.scrollWidth; // Largeur totale du texte

    if (textWidth > wrapperWidth) { // Défile uniquement si le texte dépasse
        const distance = textWidth - wrapperWidth; // Distance à parcourir
        const duration = distance / 50; // Vitesse de défilement

        element.style.transition = `transform ${duration}s linear`; // Animation fluide
        element.style.transform = `translateX(-${distance}px)`; // Défile vers la gauche
    }
}

// Fonction pour arrêter et réinitialiser le défilement
function stopScroll(element) {
    element.style.transition = 'transform 0.2s linear'; // Retour fluide
    element.style.transform = 'translateX(0)'; // Revient à la position initiale
}

function toggleFolder(element, event) {
    const folderContents = element.nextElementSibling;
    const icon = element.querySelector('i');
    if (folderContents.style.display === 'none') {
        folderContents.style.display = 'block';
        icon.classList.replace('fa-folder', 'fa-folder-open');
    } else {
        folderContents.style.display = 'none';
        icon.classList.replace('fa-folder-open', 'fa-folder');
    }
}


// Fonction pour réinitialiser le zoom de l'élément de prévisualisation
function resetPreviewZoom() {
    const previewElement = document.getElementById('preview');
    if (previewElement) {
        const canvas = previewElement.querySelector('canvas');
        if (canvas) {
            canvas.removeAttribute('data-scale'); // Supprime l'attribut de l'échelle actuelle
        }
        previewElement.style.transform = 'scale(1)'; // Réinitialise le zoom
    }
}


// Ajout de PDF.js pour l'affichage des PDF avec toutes les fonctionnalités
document.addEventListener("DOMContentLoaded", function () {
    const previewContainer = document.getElementById("preview");
    if (previewContainer) {
        previewContainer.innerHTML = ""; // Nettoyage initial
    }
});

async function openFile(filePath) {
    resetPreviewZoom();
    
    // Mise à jour des informations sur le fichier sélectionné
    const selectedFileContainer = document.getElementById('selectedFileContainer');
    const selectedFileName = document.getElementById('selectedFileName');
    const selectedFileIcon = document.getElementById('selectedFileIcon');
    
    const fileName = filePath.split('/').pop();
    const fileExtension = fileName.split('.').pop().toLowerCase();
    let iconClass = 'fas fa-file';
    switch (fileExtension) {
        case 'pdf':
            iconClass = 'fas fa-file-pdf';
            break;
        case 'docx':
        case 'doc':
            iconClass = 'fas fa-file-word';
            break;
        case 'xlsx':
        case 'xls':
            iconClass = 'fas fa-file-excel';
            break;
        case 'png':
        case 'jpg':
        case 'jpeg':
            iconClass = 'fas fa-file-image';
            break;
        case 'txt':
        case 'md':
            iconClass = 'fas fa-file-alt';
            break;
        case 'zip':
        case 'rar':
            iconClass = 'fas fa-file-archive';
            break;
        default:
            iconClass = 'fas fa-file';
    }

    selectedFileIcon.className = iconClass;
    selectedFileName.textContent = fileName;
    selectedFileContainer.style.display = 'flex';
    
    const previewContainer = document.getElementById("preview");
    previewContainer.innerHTML = "";
    const spinner = document.getElementById("loadingSpinner");
    spinner.style.display = "block";

    const relativeFilePath = filePath.startsWith("files/") ? filePath : `files/${filePath}`.replace(/\\/g, "/");

    try {
        console.log(`Ouverture du fichier : ${relativeFilePath}`);
        
        highlightSelectedFile(relativeFilePath);
        
        const extension = relativeFilePath.split(".").pop().toLowerCase();
        switch (extension) {
            case "pdf":
                const content = await window.electronAPI.readFile(relativeFilePath, { asArrayBuffer: true });
                await previewPDF(content, previewContainer);
                break;
            case 'xlsx':
            case 'xls':
            case 'xlsm':
                try {
                    const excelContent = await window.electronAPI.readFile(relativeFilePath, { asArrayBuffer: true });
                    if (!excelContent) {
                        throw new Error("Le fichier Excel n'a pas été correctement chargé.");
                    }
                    console.log("Contenu du fichier Excel chargé :", excelContent); // Debugging
                    await previewExcel(excelContent, previewContainer);
                } catch (error) {
                    console.error("Erreur lors de la lecture du fichier Excel :", error);
                    previewContainer.innerHTML = "Erreur : Impossible de charger ce fichier Excel.";
                }
                break;
            case 'png':
            case 'jpg':
            case 'jpeg':
            case 'bmp':
            case 'gif':
                try {
                    const imageContent = await window.electronAPI.readFile(relativeFilePath, { asArrayBuffer: true });
                    if (!imageContent) {
                        throw new Error("Le fichier image n'a pas été correctement chargé.");
                    }
                    console.log("Contenu du fichier image chargé :", imageContent); // Debugging
                    previewImage(imageContent, previewContainer);
                } catch (error) {
                    console.error("Erreur lors de la lecture du fichier image :", error);
                    previewContainer.innerHTML = "Erreur : Impossible de charger cette image.";
                }
                break;
            case 'docx':
            case 'doc':
                try {
                    const docxContent = await window.electronAPI.readFile(relativeFilePath, { asArrayBuffer: true });
                    if (!docxContent) {
                        throw new Error("Le fichier DOC/DOCX n'a pas été correctement chargé.");
                    }
                    console.log("Contenu du fichier DOC/DOCX chargé :", docxContent); // Debugging
                    await previewDocx(docxContent, previewContainer);
                } catch (error) {
                    console.error("Erreur lors de la lecture du fichier DOC/DOCX :", error);
                    previewContainer.innerHTML = "Erreur : Impossible de charger ce fichier DOC/DOCX.";
                }
                break;
            case 'svg':
                try {
                    const svgContent = await window.electronAPI.readFile(relativeFilePath, { asArrayBuffer: false });
                    if (!svgContent || typeof svgContent !== "string") {
                        throw new Error("Le fichier SVG n'a pas été correctement chargé.");
                    }
                    console.log("Contenu du fichier SVG chargé :", svgContent); // Debugging
                    previewSVG(svgContent, previewContainer);
                } catch (error) {
                    console.error("Erreur lors de la lecture du fichier SVG :", error);
                    previewContainer.innerHTML = "Erreur : Impossible de charger ce fichier SVG.";
                }
                break;
            case 'html':
                try {
                    const htmlContent = await window.electronAPI.readFile(relativeFilePath, { asArrayBuffer: false });
                    if (!htmlContent || typeof htmlContent !== 'string') {
                        throw new Error("Le fichier HTML n'a pas été correctement chargé.");
                    }
                    console.log("Contenu du fichier HTML chargé :", htmlContent); // Debugging
                    previewHTML(htmlContent, previewContainer);
                } catch (error) {
                    console.error("Erreur lors de la lecture du fichier HTML :", error);
                    previewContainer.innerHTML = "Erreur : Impossible de charger ce fichier HTML.";
                }
                break;
            case 'odt':
                try {
                    const odtContent = await window.electronAPI.readFile(relativeFilePath, { asArrayBuffer: true });
                    if (!odtContent) {
                        throw new Error("Le fichier ODT n'a pas été correctement chargé.");
                    }
                    console.log("Contenu du fichier ODT chargé :", odtContent); // Debugging
                    previewODT(odtContent, previewContainer);
                } catch (error) {
                    console.error("Erreur lors de la lecture du fichier ODT :", error);
                    previewContainer.innerHTML = "Erreur : Impossible de charger ce fichier ODT.";
                }
                break;
            case 'sh':
            case 'bat':
            case 'cmd':
            case 'py':
            case 'js':
            case 'ps1':
                try {
                    const scriptContent = await window.electronAPI.readFile(relativeFilePath, { asArrayBuffer: false });
                    if (!scriptContent || typeof scriptContent !== "string") {
                        throw new Error("Le fichier script n'a pas été correctement chargé.");
                    }
                    console.log("Contenu du fichier script chargé :", scriptContent); // Debugging
                    previewScript(scriptContent, previewContainer);
                } catch (error) {
                    console.error("Erreur lors de la lecture du fichier script :", error);
                    previewContainer.innerHTML = "Erreur : Impossible de charger ce script.";
                }
                break;
            default:
                const textContent = await window.electronAPI.readFile(relativeFilePath, { asArrayBuffer: true });
                await previewText(textContent, previewContainer);
        }
        await loadFileInfo(relativeFilePath);
        currentFile = relativeFilePath;
    } catch (error) {
        console.error("Erreur lors de l'ouverture du fichier : ", error);
        previewContainer.innerHTML = "Erreur lors de l'ouverture du fichier. Format non pris en charge.";
    } finally {
        spinner.style.display = "none";
    }
}

// Mise en surbrillance du fichier sélectionné par la recherche
function highlightSelectedFile(filePath) {
    const fileElements = document.querySelectorAll('.file');

    // Retire la classe 'selected' de tous les fichiers
    fileElements.forEach(fileElement => {
        fileElement.classList.remove('selected');
    });

    // Trouve le fichier correspondant au chemin
    const selectedFileElement = Array.from(fileElements).find(fileElement =>
        fileElement.getAttribute('data-path') === filePath
    );

    if (selectedFileElement) {
        // Ajoute la classe "selected"
        selectedFileElement.classList.add('selected');

        // Déplie tous les dossiers parents pour que le fichier soit visible
        let parent = selectedFileElement.parentElement;
        while (parent && parent.id !== "tree") {
            if (parent.classList.contains('folder-contents')) {
                parent.style.display = "block"; // Rendre visible
                const toggleElement = parent.previousElementSibling;
                if (toggleElement && toggleElement.classList.contains('folder-toggle')) {
                    toggleElement.querySelector('i').classList.replace('fa-folder', 'fa-folder-open');
                }
            }
            parent = parent.parentElement;
        }

        // Centrer l'élément sélectionné dans la vue
        selectedFileElement.scrollIntoView({ behavior: 'smooth', block: 'center' });
    } else {
        console.error("Élément de fichier non trouvé pour :", filePath);
    }
}


function openContainingFolder() {
    let filePath = document.getElementById('infoForm').getAttribute('data-file-path');
    if (!filePath) {
        alert("Aucun fichier sélectionné.");
        return;
    }

    // Ajoute 'files/' au chemin si c'est pas déjà le cas
    if (!filePath.startsWith('files/')) {
        filePath = `files/${filePath}`;
    }

    // Ouvre le dossier contenant le fichier sélectionné
    window.electronAPI.openContainingFolder(filePath);
}


// Fonctions de prévisualisation
function previewText(content, container) {
    const pre = document.createElement('pre');
    pre.classList.add('highlighted-script'); // Applique une mise en forme spécifique pour les scripts
    const textContent = new TextDecoder("utf-8").decode(content);
    pre.textContent = textContent;
    container.appendChild(pre);
}

function previewSVG(content, container) {
    if (!content || typeof content !== "string") {
        console.error("Le contenu du fichier SVG est invalide :", content);
        container.innerHTML = "Erreur : Contenu du fichier SVG invalide.";
        return;
    }

    console.log("Affichage du fichier SVG :", content);

    container.innerHTML = ''; // Nettoie le conteneur avant d'afficher le SVG

    const svgWrapper = document.createElement("div");
    svgWrapper.classList.add("svg-container");
    svgWrapper.innerHTML = content; // Injecte le SVG directement dans le HTML

    container.appendChild(svgWrapper);
}

function previewHTML(content, container) {
    if (!content || typeof content !== "string") {
        console.error("Le contenu HTML reçu est invalide :", content);
        container.innerHTML = "Erreur : Contenu HTML invalide.";
        return;
    }

    console.log("Affichage du contenu HTML :", content);

    const iframe = document.createElement('iframe');
    iframe.style.width = '100%';
    iframe.style.height = '100%';
    iframe.style.minHeight = '500px';
    iframe.style.border = 'none';
    iframe.style.flex = '1';
    iframe.style.resize = 'vertical';
    iframe.style.overflow = 'auto';
    iframe.sandbox = "allow-scripts allow-same-origin"; // Permet les scripts et styles internes

    container.innerHTML = ''; 
    container.appendChild(iframe);

    iframe.srcdoc = content;
}

async function previewPDF(content, container) {
    const pdfjsLib = window.pdfjsLib;
    if (!pdfjsLib) {
        console.error("pdfjsLib n'est pas chargé");
        container.innerHTML = "Erreur : pdf.js n'est pas disponible.";
        return;
    }

    const pdf = await pdfjsLib.getDocument({ data: new Uint8Array(content) }).promise;
    container.innerHTML = "";

    const pdfViewer = document.createElement("div");
    pdfViewer.classList.add("pdf-viewer");
    pdfViewer.style.overflow = "auto";
    pdfViewer.style.cursor = "grab";
    pdfViewer.addEventListener("mousedown", startDrag);
    pdfViewer.addEventListener("mouseup", stopDrag);
    pdfViewer.addEventListener("mousemove", drag);
    container.appendChild(pdfViewer);

    let scale = 1.5; // Meilleure netteté
    let currentPage = 1;
    let totalPages = pdf.numPages; // Assure que totalPages est défini avant le sélecteur

    // Vérifie si #page-info existe, sinon l'ajoute
    let pageInfoContainer = document.getElementById("page-info");
    if (!pageInfoContainer) {
        pageInfoContainer = document.createElement("div");
        pageInfoContainer.id = "page-info";
        pageInfoContainer.style.display = "flex";
        pageInfoContainer.style.alignItems = "center";
        pageInfoContainer.style.marginBottom = "10px";
        container.insertBefore(pageInfoContainer, pdfViewer);
    }

    // ✅ Création du sélecteur de pages (liste déroulante)
    const pageSelectContainer = document.createElement('div');
    pageSelectContainer.style.marginLeft = "10px";

    const pageSelect = document.createElement('select');
    pageSelect.id = 'pdf-page-select';
    pageSelect.style.padding = "5px";
    pageSelect.style.fontSize = "14px";

    for (let i = 1; i <= totalPages; i++) {
        const option = document.createElement('option');
        option.value = i;
        option.textContent = `Page ${i}`;
        pageSelect.appendChild(option);
    }

    pageSelect.addEventListener("change", () => {
        currentPage = parseInt(pageSelect.value);
        renderPage(currentPage);
    });

    pageSelectContainer.appendChild(pageSelect);
    pageInfoContainer.appendChild(pageSelectContainer);

    async function renderPage(pageNumber) {
        const page = await pdf.getPage(pageNumber);
        const viewport = page.getViewport({ scale });
        const canvas = document.createElement("canvas");
        const context = canvas.getContext("2d");
        const ratio = window.devicePixelRatio || 1;
        canvas.width = viewport.width * ratio;
        canvas.height = viewport.height * ratio;
        canvas.style.width = `${viewport.width}px`;
        canvas.style.height = `${viewport.height}px`;
        context.setTransform(ratio, 0, 0, ratio, 0, 0);
        pdfViewer.innerHTML = "";
        pdfViewer.appendChild(canvas);
        await page.render({ canvasContext: context, viewport }).promise;

        const pageInfo = document.getElementById("page-info");
        if (pageInfo) {
            const pageText = pageInfo.querySelector(".page-text");
            if (!pageText) {
                const newText = document.createElement("span");
                newText.classList.add("page-text");
                newText.textContent = `Page ${currentPage} / ${totalPages}`;
                pageInfo.insertBefore(newText, pageInfo.firstChild);
            } else {
                pageText.textContent = `Page ${currentPage} / ${totalPages}`;
            }
        }
        pageSelect.value = currentPage;
    }

    function updateNavigationButtons() {
        const previousButton = document.getElementById('previous-page-button');
        const nextButton = document.getElementById('next-page-button');
        previousButton.disabled = currentPage <= 1;
        nextButton.disabled = currentPage >= totalPages;
    }

    function startDrag(e) {
        isDragging = true;
        startX = e.clientX;
        startY = e.clientY;
        scrollLeft = pdfViewer.scrollLeft;
        scrollTop = pdfViewer.scrollTop;
        pdfViewer.style.cursor = "grabbing";
    }

    function stopDrag() {
        isDragging = false;
        pdfViewer.style.cursor = "grab";
    }

    function drag(e) {
        if (!isDragging) return;
        e.preventDefault();
        pdfViewer.scrollLeft = scrollLeft - (e.clientX - startX);
        pdfViewer.scrollTop = scrollTop - (e.clientY - startY);
    }

    document.getElementById('previous-page-button').addEventListener('click', () => {
        if (currentPage > 1) {
            currentPage--;
            renderPage(currentPage);
        }
    });

    document.getElementById('next-page-button').addEventListener('click', () => {
        if (currentPage < totalPages) {
            currentPage++;
            renderPage(currentPage);
        }
    });

    renderPage(currentPage);
}


function previewScript(content, container) {
    if (!content || typeof content !== "string") {
        console.error("Le contenu du script reçu est invalide :", content);
        container.innerHTML = "Erreur : Contenu du script invalide.";
        return;
    }

    console.log("Affichage du script :", content);

    const pre = document.createElement('pre');
    pre.classList.add('script-preview');

    // Détection du langage (JS, Python, Shell, Batch, PowerShell)
    let langClass = 'language-none'; // Par défaut sans langage spécifique
    if (container.dataset.fileExtension === "js") langClass = "language-javascript";
    else if (container.dataset.fileExtension === "py") langClass = "language-python";
    else if (container.dataset.fileExtension === "sh") langClass = "language-bash";
    else if (container.dataset.fileExtension === "bat" || container.dataset.fileExtension === "cmd") langClass = "language-batch";
    else if (container.dataset.fileExtension === "ps1") langClass = "language-powershell"; // Ajout PowerShell

    // Ajout du contenu du script dans un bloc `<code>` avec la bonne classe Prism.js
    const code = document.createElement('code');
    code.classList.add(langClass);
    code.textContent = content;

    pre.appendChild(code);
    container.innerHTML = '';
    container.appendChild(pre);

    // Appliquer Prism.js pour la coloration syntaxique
    if (window.Prism) {
        Prism.highlightElement(code);
    }
}


async function previewExcel(content, container) {
    if (!content || !(content instanceof ArrayBuffer)) {
        console.error("Le contenu du fichier Excel est invalide :", content);
        container.innerHTML = "Erreur : Contenu du fichier Excel invalide.";
        return;
    }

    console.log("Affichage du fichier Excel :", content);

    try {
        // Chargement du fichier avec SheetJS
        const workbook = XLSX.read(content, { type: 'array' });

        const tabContainer = document.createElement('div');
        tabContainer.classList.add('tab-container');

        const sheetContainer = document.createElement('div');
        sheetContainer.classList.add('sheet-container');

        container.innerHTML = ''; // Nettoie le conteneur
        container.appendChild(tabContainer);
        container.appendChild(sheetContainer);

        // Création des onglets pour les différentes feuilles du fichier Excel
        workbook.SheetNames.forEach((sheetName, index) => {
            const tab = document.createElement('button');
            tab.classList.add('tab');
            tab.textContent = sheetName;
            tab.addEventListener('click', () => {
                displaySheet(sheetName, workbook, sheetContainer);
            });

            tabContainer.appendChild(tab);

            if (index === 0) {
                displaySheet(sheetName, workbook, sheetContainer);
                tab.classList.add('active');
            }
        });
    } catch (error) {
        console.error("Erreur lors de la prévisualisation du fichier Excel :", error);
        container.innerHTML = "Erreur lors de l'ouverture du fichier. Format non pris en charge.";
    }
}

function displaySheet(sheetName, workbook, container) {
    if (!container) {
        console.error("Conteneur pour la feuille Excel non trouvé.");
        return;
    }

    container.innerHTML = '<div class="spinner">Chargement...</div>';

    setTimeout(() => {
        container.innerHTML = '';
        const worksheet = workbook.Sheets[sheetName];
        const htmlContent = XLSX.utils.sheet_to_html(worksheet);
        container.innerHTML = htmlContent;

        const tabsContainer = container.previousElementSibling;
        if (tabsContainer) {
            const tabs = tabsContainer.querySelectorAll('.tab');
            tabs.forEach(tab => {
                tab.classList.remove('active');
                if (tab.textContent === sheetName) {
                    tab.classList.add('active');
                }
            });
        } else {
            console.error("Onglets non trouvés.");
        }
    }, 500);
}

// Preview des fichiers images
function previewImage(content, container) {
    if (!content || !(content instanceof ArrayBuffer)) {
        console.error("Le contenu de l'image reçu est invalide :", content);
        container.innerHTML = "Erreur : Contenu d'image invalide.";
        return;
    }

    console.log("Affichage de l'image :", content);

    const blob = new Blob([content]);
    const imgURL = URL.createObjectURL(blob);

    const img = document.createElement('img');
    img.src = imgURL;
    img.style.maxWidth = '100%';
    img.style.height = 'auto';
    img.style.display = 'block';
    img.style.margin = 'auto';

    container.innerHTML = '';
    container.appendChild(img);
}

// Prévisualisation des fichiers de type docx
async function previewDocx(content, container) {
    if (!content || !(content instanceof ArrayBuffer)) {
        console.error("Le contenu du fichier DOC/DOCX est invalide :", content);
        container.innerHTML = "Erreur : Contenu du fichier DOC/DOCX invalide.";
        return;
    }

    console.log("Affichage du fichier DOC/DOCX :", content);

    try {
        // Conversion du fichier avec Mammoth.js
        const result = await mammoth.convertToHtml({ arrayBuffer: content });
        const html = result.value; // Contenu HTML extrait du DOCX
        const messages = result.messages; // Messages d'erreurs éventuels

        if (messages && messages.length > 0) {
            console.warn("Messages de conversion:", messages);
        }

        // Création du conteneur ajusté
        const docContainer = document.createElement("div");
        docContainer.classList.add("doc-preview");
        docContainer.innerHTML = html;

        // Ajout du conteneur dans la prévisualisation
        container.innerHTML = "";
        container.appendChild(docContainer);
    } catch (error) {
        console.error("Erreur lors de la conversion du fichier DOCX :", error);
        container.innerHTML = "Erreur : Impossible de lire ce document Word.";
    }
}

// Preview des fichiers MD
function previewMarkdown(content, container) {
    const markdownContent = new TextDecoder().decode(content);
    const converter = new showdown.Converter();
    const htmlContent = converter.makeHtml(markdownContent);
    container.innerHTML = htmlContent;

    updateNavigationButtons(1, 1); // MàJ des boutons de navigation pour les MD
}

// Preview des fichiers ODT
function previewODT(content, container) {
    if (!content || !(content instanceof ArrayBuffer)) {
        console.error("Le contenu du fichier ODT est invalide :", content);
        container.innerHTML = "Erreur : Contenu du fichier ODT invalide.";
        return;
    }

    console.log("Affichage du fichier ODT :", content);

    // Crée un `Blob` pour charger le fichier ODT
    const blob = new Blob([content], { type: "application/vnd.oasis.opendocument.text" });
    const url = URL.createObjectURL(blob);

    // Crée un conteneur pour afficher le document
    const odfContainer = document.createElement("div");
    odfContainer.id = "odf-container";

    // S'assure que le conteneur est bien vide avant d'ajouter l'affichage
    container.innerHTML = ''; 
    container.appendChild(odfContainer);

    // Chargement du fichier avec WebODF
    const odfViewer = new odf.OdfCanvas(odfContainer);
    odfViewer.load(url);

    // Nettoyage du Blob après affichage
    odfViewer.addListener("statereadychange", function () {
        URL.revokeObjectURL(url);
    });
}

//fonctions de tags (ajouté à la version 1.2.1)
// ********************************************************* //
// Fonction pour créer un élément de tag
function createTagElement(tag) {
    const span = document.createElement('span');
    span.className = 'tag'; // Classe CSS pour les tags
    span.textContent = tag.trim(); // on fait gaffe de ne pas ajouter d'espace ou de texte indésirable

    const closeButton = document.createElement('button');
    closeButton.className = 'close-btn';
    closeButton.textContent = '×';

    // Suppression du tag au clic
    closeButton.addEventListener('click', () => {
        span.remove();
        console.log(`Tag supprimé : "${tag}"`);
    });

    span.appendChild(closeButton);
    return span;
}


// Fonction pour réinitialiser les tags affichés
function resetTags() {
    const tagsContainer = document.getElementById('fileTags');
    if (!tagsContainer) {
        console.error("Conteneur de tags introuvable pour réinitialisation.");
        return;
    }

    console.log('Réinitialisation des tags déclenchée. Avant :', Array.from(tagsContainer.children).map(el => el.textContent));
    tagsContainer.innerHTML = ''; // Vide tous les tags
    console.log('Après réinitialisation :', Array.from(tagsContainer.children));
}

// Fonction pour supprimer un tag avec une animation
function removeTag(tagElement) {
    console.log(`Suppression du tag déclenchée depuis : ${tagElement.outerHTML}`);
    tagElement.classList.add('hide'); // Ajoute une animation de suppression
    tagElement.addEventListener('transitionend', () => {
        console.log(`Tag supprimé : ${tagElement.textContent.trim()}`);
        tagElement.remove(); // Supprime l'élément après l'animation
    });
}

function isDuplicate(tagValue, tagsContainer) {
    return Array.from(tagsContainer.querySelectorAll('.tag')).some(
        (tagElement) =>
            tagElement.textContent.replace('×', '').trim() === tagValue.trim()
    );
}

// Gestionnaire de chargement des tags
async function loadFileInfo(filePath) {
    const relativeFilePath = filePath.startsWith('files/') || filePath.startsWith('files\\')
        ? filePath
        : `files/${filePath}`.replace(/\\/g, '/');

    const jsonFilePath = `${relativeFilePath}.json`; // Génère le chemin du fichier JSON
    console.log(`Chargement du fichier JSON : ${jsonFilePath}`);

    try {
        const fileInfo = await window.electronAPI.readJsonFile(jsonFilePath) || {};
        console.log('Contenu du fichier JSON chargé :', fileInfo);

        // Mise à jour des champs du formulaire
        document.getElementById('fileDate').value = fileInfo.date || '';
        document.getElementById('fileVersion').value = fileInfo.version || '';
        document.getElementById('fileAuthor').value = fileInfo.author || '';
        document.getElementById('fileComments').value = fileInfo.comments || '';
        document.getElementById('infoForm').setAttribute('data-file-path', filePath);

        // Réinitialiser les tags
        resetTags();

        // Ajouter les tags du fichier JSON
        if (fileInfo.tags && Array.isArray(fileInfo.tags)) {
            fileInfo.tags.forEach(tag => {
                if (tag) {
                    console.log(`Ajout du tag : ${tag}`);
                    const tagElement = createTagElement(tag);
                    document.getElementById('fileTags').appendChild(tagElement);

                    // Ajouter la classe 'show' pour déclencher l'animation
                    requestAnimationFrame(() => {
                        tagElement.classList.add('show');
                    });
                }
            });
        } else {
            console.log('Aucun tag à afficher.');
        }

        console.log('Tags affichés après chargement :', Array.from(document.getElementById('fileTags').children).map(tagElement => tagElement.textContent));
    } catch (error) {
        console.error(`Erreur lors du chargement du fichier JSON : ${jsonFilePath}`, error);
    }
}

// Gestionnaire principal
document.addEventListener('DOMContentLoaded', () => {
    const inputField = document.getElementById('fileTagsInput');
    const addTagButton = document.getElementById('addTagButton');
    const tagsContainer = document.getElementById('fileTags');

    // Vérifiez si les éléments existent
    if (!inputField || !addTagButton || !tagsContainer) {
        console.error("Un ou plusieurs éléments nécessaires sont introuvables !");
        return;
    }

    // Gestionnaire d'événement pour le bouton
    addTagButton.addEventListener('click', () => {
        const tagValue = inputField.value.trim();
        console.log(`Valeur du champ d'entrée : "${tagValue}"`); // Vérifiez la valeur saisie

        if (!tagValue) {
            console.warn("Le champ de saisie est vide.");
            return;
        }

        if (isDuplicate(tagValue, tagsContainer)) {
            console.warn(`Le tag "${tagValue}" existe déjà.`);
            return;
        }

        // Créez un nouveau tag
        const tagElement = createTagElement(tagValue);

        // Ajoutez-le au conteneur
        tagsContainer.appendChild(tagElement);

        // Ajoutez l'animation après l'insertion dans le DOM
        requestAnimationFrame(() => {
            tagElement.classList.add('show');
        });

        // Réinitialiser le champ d'entrée
        inputField.value = '';
        console.log(`Tag ajouté manuellement : "${tagValue}"`);
    });

    // Gestionnaire pour la touche Entrée
    inputField.addEventListener('keypress', (event) => {
        if (event.key === 'Enter') {
            event.preventDefault(); // Empêche le comportement par défaut
            addTagButton.click(); // Simule un clic sur le bouton
        }
    });
});

  // Fonction pour sauvegarder les informations associées à un fichier
  function saveFileInfo() {
    const filePath = document.getElementById('infoForm').getAttribute('data-file-path');
    if (!filePath) {
        // Utiliser une traduction pour le message d'alerte
        const noFileMessage = translations["no_file_selected"] || "Aucun fichier sélectionné.";
        alert(noFileMessage);
        return;
    }

    const relativeFilePath = filePath.startsWith('files/') || filePath.startsWith('files\\')
        ? filePath
        : `files/${filePath}`.replace(/\\/g, '/');

    const jsonFilePath = `${relativeFilePath}.json`; // Utilisation du chemin corrigé
    console.log(`Sauvegarde dans le fichier JSON : ${jsonFilePath}`);

    const fileInfo = {
        date: document.getElementById('fileDate').value,
        version: document.getElementById('fileVersion').value,
        author: document.getElementById('fileAuthor').value,
        comments: document.getElementById('fileComments').value,
        tags: Array.from(document.querySelectorAll('#fileTags .tag')).map(tagElement =>
            tagElement.firstChild.textContent.trim()
        )
    };

    window.electronAPI.writeJsonFile(jsonFilePath, fileInfo);

    // Utiliser une traduction pour le message de confirmation
    const successMessage = translations["save_success_message"] || "Informations sauvegardées avec succès.";
    showAlertMessage(successMessage, 'success');
}

// -*---*-*-*-*-*----***-*-

function displaySearchResults(results, searchTerm) {
    resetTreeView(); // Réinitialise l'arborescence avant chaque recherche

    const treeDiv = document.getElementById('tree');
    const items = treeDiv.getElementsByTagName('li');

    results.forEach(result => {
        const { file } = result;

        console.log("Recherche dans le DOM pour le fichier :", file);

        // Rechercher l'élément correspondant dans le DOM
        const item = Array.from(items).find(
            el => el.getAttribute('onclick') && el.getAttribute('onclick').includes(file)
        );

        if (item) {
            console.log("Élément trouvé dans le DOM :", item);

            // Surligne l'élément trouvé
            item.classList.add('highlight');
            item.style.display = ""; // visible

            // Déplie les dossiers parents pour afficher le fichier
            let parent = item.parentElement;
            while (parent && parent.className !== "tree") {
                if (parent.classList.contains('folder-contents')) {
                    parent.style.display = "block";
                    const toggleElement = parent.previousElementSibling;
                    if (toggleElement && toggleElement.classList.contains('folder-toggle')) {
                        toggleElement.querySelector('i').classList.replace('fa-folder', 'fa-folder-open');
                    }
                }
                parent = parent.parentElement;
            }

            // Ajouter une indication visuelle de la correspondance
            const matchIndicator = document.createElement('span');
            matchIndicator.className = 'match-indicator';
            matchIndicator.innerHTML = `Résultat trouvé : ${determineMatchType(result, searchTerm)}`;
            item.appendChild(matchIndicator);
        } else {
            console.log("Élément non trouvé pour le fichier :", file);
        }
    });
}


function determineMatchType(result, searchTerm) {
    const { file, tags = [], content = "" } = result; // Défaut pour éviter les erreurs si `tags` ou `content` sont absents
    const term = searchTerm.toLowerCase();

    if (file.toLowerCase().includes(term)) {
        return "Nom du fichier";
    } else if (Array.isArray(tags) && tags.some(tag => tag.toLowerCase().includes(term))) {
        return "Tag";
    } else if (content.toLowerCase().includes(term)) {
        return "Contenu";
    }
    return "";
}

function extractExcerpt(content = "", searchTerm) {
    const term = searchTerm.toLowerCase();
    const startIdx = content.toLowerCase().indexOf(term);

    if (startIdx === -1) return ""; // Pas de correspondance dans le contenu

    const snippetLength = 30;
    const start = Math.max(0, startIdx - snippetLength);
    const end = Math.min(content.length, startIdx + snippetLength);

    const snippet = content.substring(start, end);
    return `...${snippet.replace(new RegExp(term, 'gi'), `<span class="highlight">${searchTerm}</span>`)}...`;
}

// mise en surbrillance des fichiers dans l'arborescence après recherche
function highlightFileInTree(filePath) {
    const treeDiv = document.getElementById('tree');
    const items = treeDiv.getElementsByTagName('li');

    // Normaliser le chemin
    const normalizedFilePath = filePath.startsWith('files/') ? filePath : `files/${filePath}`;

    // Réinitialiser la surbrillance
    Array.from(items).forEach(item => {
        item.classList.remove('highlight');
    });

    // Trouver l'élément correspondant
    const targetItem = Array.from(items).find(
        el => el.getAttribute('data-path') === normalizedFilePath
    );

    if (targetItem) {
        // Surligner et déplier les dossiers parents
        targetItem.classList.add('highlight');
        let parent = targetItem.parentElement;
        while (parent && parent.id !== "tree") {
            if (parent.classList.contains('folder-contents')) {
                parent.style.display = "block";
                const toggleElement = parent.previousElementSibling;
                if (toggleElement && toggleElement.classList.contains('folder-toggle')) {
                    toggleElement.querySelector('i').classList.replace('fa-folder', 'fa-folder-open');
                }
            }
            parent = parent.parentElement;
        }

        // Scroller vers l'élément pour le rendre visible
        targetItem.scrollIntoView({ behavior: 'smooth', block: 'center' });
    } else {
        console.error(`Élément non trouvé pour le fichier : ${normalizedFilePath}`);
    }
}


// Fonction pour réinitialiser l'arborescence des fichiers
function resetTreeView() {
    console.log("Réinitialisation de l'arborescence.");
    const treeDiv = document.getElementById('tree');
    if (!treeDiv) {
        console.error("Élément #tree introuvable pour la réinitialisation !");
        return;
    }

    const items = treeDiv.getElementsByTagName('li');
    for (let i = 0; i < items.length; i++) {
        items[i].style.display = ""; // Affiche tous les éléments
        items[i].classList.remove('highlight'); // Supprime la surbrillance
    }

    const folderContents = treeDiv.getElementsByClassName('folder-contents');
    for (let i = 0; i < folderContents.length; i++) {
        folderContents[i].style.display = 'none';
    }

    const folderToggles = treeDiv.getElementsByClassName('folder-toggle');
    for (let i = 0; i < folderToggles.length; i++) {
        const icon = folderToggles[i].querySelector('i');
        if (icon && icon.classList.contains('fa-folder-open')) {
            icon.classList.replace('fa-folder-open', 'fa-folder');
        }
    }
    console.log("Réinitialisation de l'arborescence terminée.");
}


// Fonction pour afficher un message d'alerte
async function performSearch(query) {
    const modalSearchResultsList = document.getElementById('modalSearchResultsList');
    console.log("Recherche lancée avec la requête :", query);

    if (!query.trim()) {
        modalSearchResultsList.innerHTML = `
            <li>
                <div class="scrollable-wrapper">
                    <span class="scrollable">${translations["search_no_results"]}</span>
                </div>
            </li>`;
        return;
    }

    try {
        modalSearchResultsList.innerHTML = `
            <li>
                <div class="scrollable-wrapper">
                    <span class="scrollable">${translations["search_in_progress"]}</span>
                </div>
            </li>`;

        // Diviser la requête en mots-clés
        const keywords = query.trim().toLowerCase().split(/\s+/);
        console.log("Mots-clés analysés :", keywords);

        // Récupérer les résultats pour chaque mot-clé
        const allResults = await Promise.all(
            keywords.map(async (keyword) => {
                const results = await window.electronAPI.searchFiles(keyword);
                return results.map(result => ({ ...result, matchedKeyword: keyword }));
            })
        );

        // Combiner les résultats et collecter les correspondances de mots-clés
        const fileMap = {};
        allResults.flat().forEach(result => {
            const file = result.file;
            if (!fileMap[file]) {
                fileMap[file] = { 
                    file, 
                    matchedKeywords: new Set(), 
                    tags: result.tags, 
                    content: result.content 
                };
            }
            fileMap[file].matchedKeywords.add(result.matchedKeyword);
        });

        // Classer les fichiers en fonction du nombre de mots-clés correspondants
        const sortedFiles = Object.values(fileMap).sort((a, b) => b.matchedKeywords.size - a.matchedKeywords.size);

        modalSearchResultsList.innerHTML = '';

        if (sortedFiles.length === 0) {
            modalSearchResultsList.innerHTML = `
                <li>
                    <div class="scrollable-wrapper">
                        <span class="scrollable">${translations["search_no_files"]}</span>
                    </div>
                </li>`;
        } else {
            // Affichage des fichiers classés par pertinence
            sortedFiles.forEach(({ file, matchedKeywords }) => {
                const listItem = document.createElement('li');
                listItem.classList.add('search-result-item');

                const relativeFilePath = file.replace(/^.*\/files\//, 'files/');
                const iconClass = getFileIconClass(relativeFilePath);

                const iconElement = document.createElement('i');
                iconElement.className = iconClass;

                const scrollableWrapper = document.createElement('div');
                scrollableWrapper.className = 'scrollable-wrapper';

                const scrollableText = document.createElement('span');
                scrollableText.className = 'scrollable';
                scrollableText.innerHTML = `
                    ${relativeFilePath} <br>
                    <small>
                        <b>${translations["search_keywords_found"]} :</b> ${Array.from(matchedKeywords).join(', ')} <br>
                        <b>${translations["search_match"]} :</b> (${matchedKeywords.size}/${keywords.length})
                    </small>
                `;

                scrollableText.addEventListener('mouseover', () => startScroll(scrollableText));
                scrollableText.addEventListener('mouseout', () => stopScroll(scrollableText));

                scrollableWrapper.appendChild(scrollableText);
                listItem.appendChild(iconElement);
                listItem.appendChild(scrollableWrapper);

                listItem.onclick = () => {
                    openFile(relativeFilePath);
                    const modal = document.getElementById('searchModal');
                    modal.style.display = 'none';
                };

                modalSearchResultsList.appendChild(listItem);
            });
        }
    } catch (error) {
        console.error("Erreur lors de la recherche :", error);
        modalSearchResultsList.innerHTML = `
            <li>
                <div class="scrollable-wrapper">
                    <span class="scrollable">${translations["search_error"]}</span>
                </div>
            </li>`;
    }
}


// Fonction pour afficher le résultat de la recherche dans une popup
function showSearchResultsPopup() {
    const modal = document.getElementById('searchModal');
    const searchInputModal = document.getElementById('searchInputModal');
    const modalSearchResultsList = document.getElementById('modalSearchResultsList');
    const closeModal = document.getElementById('closeModal');

    // Effacer les résultats précédents et réinitialiser le champ de recherche
    modalSearchResultsList.innerHTML = `<li>${translations["search_no_results"]}</li>`;
    searchInputModal.value = '';

    // Ajoutez la classe pour un défilement horizontal
    if (!modalSearchResultsList.classList.contains('scrollable-results')) {
        modalSearchResultsList.classList.add('scrollable-results');
    }

    // Afficher la modale
    modal.style.display = 'block';
    searchInputModal.focus();

    // Rechercher en temps réel lorsque l'utilisateur tape
    searchInputModal.oninput = function () {
        performSearch(searchInputModal.value);
    };

    // Fonction de fermeture de la modale
    closeModal.onclick = () => {
        modal.style.display = 'none';
    };

    // Fermer la modale en cliquant en dehors de son contenu
    window.onclick = function (event) {
        if (event.target === modal) {
            modal.style.display = 'none';
        }
    };
}


// Fonction pour gérer le clic sur le bouton pour afficher la recherche
function toggleSearchInput() {
    showSearchResultsPopup(); // Appel de la fonction qui affiche la popup de recherche
}

// Ajouter un événement au bouton de recherche pour déclencher la modale
document.getElementById('searchButton').onclick = toggleSearchInput;

// Ajoute des écouteurs d'événements pour les dossiers au chargement de la page
document.addEventListener('DOMContentLoaded', () => {
    const treeDiv = document.getElementById('tree');
    const firstItem = treeDiv.querySelector('li');
    
    if (firstItem) {
        firstItem.classList.add('highlight');
    }

    // Ajout de l'écouteur pour bloquer la propagation des clics dans le conteneur des tags
    const tagsContainer = document.getElementById('fileTags');
    if (tagsContainer) {
        tagsContainer.addEventListener('click', function (event) {
            if (event.target.tagName !== 'BUTTON') {
                console.log("Propagation bloquée uniquement pour les clics non liés à un bouton.");
            }
        });
        console.log("Écouteur ajouté pour le conteneur #fileTags.");
    } else {
        console.warn("Conteneur #fileTags non trouvé.");
    }
});


// Fonction de saisie de la recherche
function toggleSearchInput() {
    const modal = document.getElementById('searchModal');
    const searchInput = document.getElementById('searchInputModal'); 
    const modalSearchResultsList = document.getElementById('modalSearchResultsList'); 

    // Affiche la modale
    modal.style.display = 'block';
    searchInput.value = ''; 
    searchInput.focus(); 
    modalSearchResultsList.innerHTML = `<li>${translations["search_no_results"]}</li>`; 

    // Fermer la modale
    const closeModal = document.getElementById('closeModal');
    closeModal.onclick = () => {
        modal.style.display = 'none';
        resetTreeView(); 
    };

    // Recherche dynamique
    searchInput.oninput = () => performSearch(searchInput.value);

    // Fermer la modale en cliquant en dehors de son contenu
    window.onclick = function(event) {
        if (event.target === modal) {
            modal.style.display = 'none';
            resetTreeView();
        }
    };
}

async function refreshDirectory() {
    const fileCounterElement = document.getElementById('fileCounter');
    console.log("Début du rafraîchissement de la GED...");

    // Réinitialiser l'état de l'interface utilisateur
    resetInterfaceState();

    try {
        console.log("Affichage de la boîte de dialogue de rafraîchissement.");
        const userChoice = await window.electronAPI.showRefreshDialog();

        if (userChoice === 2) {
            console.log("Rafraîchissement annulé par l'utilisateur.");
            return;
        }

        let reindexationPerformed = false;

        if (userChoice === 0) {
            console.log("Choix : Rafraîchissement avec réindexation...");
            await window.electronAPI.initializeIndexing();
            reindexationPerformed = true;
            console.log("Réindexation terminée.");
        } else {
            console.log("Choix : Rafraîchissement sans réindexation...");
        }

        console.log("Récupération des fichiers pour mise à jour de l'arborescence...");
        const data = await window.electronAPI.getFiles();
        if (data && typeof data === 'object') {
            displayDirectoryStructure(data);
            resetTreeView();
            console.log("Mise à jour de l'arborescence réussie.");
        } else {
            console.warn("Données invalides reçues pour l'arborescence.", data);
            throw new Error("Données retournées pour l’arborescence invalides.");
        }

        console.log("Mise à jour du compteur de fichiers...");
        const fileCount = await window.electronAPI.countFiles();
        updateFileCounter(fileCount);
        console.log(`Compteur de fichiers mis à jour : ${fileCount}`);

        if (reindexationPerformed) {
            console.log("Rafraîchissement additionnel après réindexation...");
            await refreshDirectoryWithoutDialog();
        }

    } catch (error) {
        console.error("Erreur lors de la mise à jour des fichiers :", error);
        if (fileCounterElement) {
            fileCounterElement.innerHTML = translations["file_counter_error"] || "Erreur de chargement des fichiers";
        }
    } finally {
        console.log("Rafraîchissement terminé.");
        updateTranslations();
    }
}

function resetInterfaceState() {
    const elementsToReset = [
        { id: 'searchInput', value: '' },
        { id: 'preview', innerHTML: '' },
        { id: 'selectedFileContainer', style: { display: 'none' } },
        { id: 'selectedFileName', textContent: translations["no_file_selected"] || 'Aucun fichier sélectionné' },
        { id: 'fileDate', value: '' },
        { id: 'fileVersion', value: '' },
        { id: 'fileAuthor', value: '' },
        { id: 'fileTagsInput', value: '' },
        { id: 'fileComments', value: '' },
        { id: 'fileTags', innerHTML: '' },
    ];

    elementsToReset.forEach(({ id, ...props }) => {
        const element = document.getElementById(id);
        if (element) {
            Object.entries(props).forEach(([key, value]) => {
                element[key] = value;
            });
        }
    });

    const alertMessage = document.getElementById('alertMessage');
    if (alertMessage) {
        alertMessage.classList.add('d-none');
    }
}

function updateFileCounter(fileCount) {
    const fileCounterElement = document.getElementById('fileCounter');
    if (fileCounterElement) {
        const counterTemplate = translations["file_counter"] || "Nb de fichiers en GED : XX";
        fileCounterElement.dataset.fileCount = fileCount;
        fileCounterElement.innerHTML = counterTemplate.replace("XX", fileCount);
    }
}

function showInfo() {
    $('#infoModal').modal('show');
}

function quitApplication() {
    window.electronAPI.quitApp();  // Appelle la fonction exposée dans preload.js
}

function openFileInNativeApp() {
    const filePath = document.getElementById('infoForm').getAttribute('data-file-path');
    if (!filePath) {
        alert("Aucun fichier sélectionné.");
        return;
    }

    // Ouvrir le fichier dans l'application native
    window.electronAPI.openFile(filePath);
}

function getFileIconClass(fileName) {
    const extension = fileName.split('.').pop().toLowerCase();
    switch (extension) {
        case 'pdf':
            return 'fas fa-file-pdf';
        case 'xlsx':
        case 'xls':
        case 'xlsm':
            return 'fas fa-file-excel';
        case 'png':
        case 'jpg':
        case 'jpeg':
        case 'bmp':
        case 'gif':
            return 'fas fa-file-image';
        case 'docx':
        case 'doc':
            return 'fas fa-file-word';
        case 'pptx':
        case 'ppt':
            return 'fas fa-file-powerpoint';
        case 'svg':
        case 'odt':
            return 'fas fa-file-alt';
        case 'txt':
        case 'md':
            return 'fas fa-file-alt';
        case 'sh':
        case 'bat':
        case 'cmd':
        case 'py':
            return 'fas fa-file-code';
        default:
            return 'fas fa-file';
    }
}

// gestion des noms de fichiers affichés en haut
document.addEventListener('DOMContentLoaded', () => {
    const navigationPane = document.getElementById('navigationPane');
    const selectedFileName = document.getElementById('selectedFileName');

    // Écoute des clics dans le panneau de navigation
    navigationPane.addEventListener('click', (event) => {
        const clickedElement = event.target.closest('.file-item');
        
        if (clickedElement) {
            const fileName = clickedElement.textContent.trim(); // Récupère le nom du fichier
            selectedFileName.textContent = fileName || 'Aucun fichier sélectionné';
        }
    });
});


window.addEventListener('resize', function () {
    if (currentFile && currentFile.endsWith('.pdf')) {
        previewPDF(lastContentBuffer, document.getElementById('preview'));
    }
});

// ajout redimmensionnement de la navigation //
const resizer = document.getElementById('resizer');
const leftPane = document.getElementById('navigationPane');
const rightPane = document.getElementById('previewPane');

let x = 0;
let leftWidth = 0;

const mouseDownHandler = function(e) {
    x = e.clientX;
    leftWidth = leftPane.getBoundingClientRect().width;

    document.addEventListener('mousemove', mouseMoveHandler);
    document.addEventListener('mouseup', mouseUpHandler);
};

const mouseMoveHandler = function(e) {
    const dx = e.clientX - x;
    const newLeftWidth = leftWidth + dx;

    if (newLeftWidth >= 50 && newLeftWidth <= 300) {
        leftPane.style.width = `${newLeftWidth}px`;
        rightPane.style.flex = '1'; 
    }
};

const mouseUpHandler = function() {
    document.removeEventListener('mousemove', mouseMoveHandler);
    document.removeEventListener('mouseup', mouseUpHandler);
};


resizer.addEventListener('mousedown', mouseDownHandler);

// Ajout des gestionnaires d'événements pour les boutons
document.addEventListener('DOMContentLoaded', (event) => {
    // Ajouter des événements pour les boutons de navigation
    const zoomInButton = document.getElementById('zoom-in-button');
    const zoomOutButton = document.getElementById('zoom-out-button');

    if (zoomInButton) {
        zoomInButton.addEventListener('click', () => {
            window.electronAPI.zoomIn(); // Envoyer l'événement au processus principal
        });
    }

    if (zoomOutButton) {
        zoomOutButton.addEventListener('click', () => {
            window.electronAPI.zoomOut(); // Envoyer l'événement au processus principal
        });
    }

    window.electronAPI.receive('zoom-in', () => {
        const previewElement = document.getElementById('preview');
        if (previewElement) {
            let currentZoom = parseFloat(previewElement.style.zoom) || 1;
            currentZoom += 0.1;
            previewElement.style.zoom = currentZoom;
        }
    });

    window.electronAPI.receive('zoom-out', () => {
        const previewElement = document.getElementById('preview');
        if (previewElement) {
            let currentZoom = parseFloat(previewElement.style.zoom) || 1;
            currentZoom = Math.max(0.1, currentZoom - 0.1);
            previewElement.style.zoom = currentZoom;
        }
    });
});


function loadDocument(url) {
    // Utiliser pdf.js pour charger le document et obtenir le nombre total de pages
    const loadingTask = pdfjsLib.getDocument(url);
    loadingTask.promise.then(pdf => {
        totalPages = pdf.numPages;
        currentPage = 1; // Réinitialisation à la première page
        updateNavigationButtons();
    }).catch(error => {
        console.error('Erreur lors du chargement du document :', error);
    });
}

document.addEventListener('DOMContentLoaded', (event) => {
    // Ajouter des événements pour les boutons de navigation
    const zoomInButton = document.getElementById('zoom-in-button');
    const zoomOutButton = document.getElementById('zoom-out-button');
    const previousPageButton = document.getElementById('previous-page-button');
    const nextPageButton = document.getElementById('next-page-button');

    // Vérification des boutons dans la console
    console.log("Bouton Zoom In", zoomInButton);
    console.log("Bouton Zoom Out", zoomOutButton);

    // Ajouter les gestionnaires d'événements aux boutons
    if (zoomInButton) {
        zoomInButton.addEventListener('click', zoomIn);
    }

    if (zoomOutButton) {
        zoomOutButton.addEventListener('click', zoomOut);
    }

    if (previousPageButton) {
        previousPageButton.addEventListener('click', previousPage);
    }

    if (nextPageButton) {
        nextPageButton.addEventListener('click', nextPage);
    }

    // Appeler updateNavigationButtons après le chargement du document
    updateNavigationButtons();

    // Fonction pour ajouter des classes et des événements aux onglets
    function handleTabs(tabs) {
        const pdfViewer = document.getElementById('pdf-viewer');
        const previewElement = document.getElementById('preview');

        tabs.forEach(tab => {
            // Ajouter dynamiquement la classe pdf-tab si l'onglet concerne un PDF
            const tabNumber = tab.getAttribute('data-tab');
            if (tabNumber === '3' || tabNumber === '4') {
                tab.classList.add('pdf-tab');
            }

            tab.addEventListener('click', () => {
                // Retirer la classe active de tous les onglets
                tabs.forEach(t => t.classList.remove('active'));
                // Ajouter la classe active à l'onglet cliqué
                tab.classList.add('active');

                // Logique pour afficher le document correspondant
                if (tabNumber === '3' || tabNumber === '4') {
                    pdfViewer.style.display = 'block';
                    previewElement.style.display = 'none';
                } else {
                    pdfViewer.style.display = 'none';
                    previewElement.style.display = 'block';
                }
            });
        });
    }

    // Initialiser les onglets existants
    handleTabs(document.querySelectorAll('.tab'));

    // Observer les changements dans le DOM pour détecter les nouveaux onglets
    const observer = new MutationObserver((mutations) => {
        mutations.forEach((mutation) => {
            if (mutation.addedNodes.length) {
                mutation.addedNodes.forEach((node) => {
                    if (node.classList && node.classList.contains('tab')) {
                        handleTabs([node]);
                    }
                });
            }
        });
    });

    // Configurer l'observateur pour surveiller les ajouts d'enfants dans le conteneur des onglets
    const tabContainer = document.querySelector('.tab-container');
    observer.observe(tabContainer, { childList: true });
});

function setActiveTab(tabButton) {
    // Retirer la classe active de tous les onglets
    const tabs = document.querySelectorAll('.pdf-tab');
    tabs.forEach(tab => {
        tab.classList.remove('active');
    });

    // Ajouter la classe active à l'onglet cliqué
    tabButton.classList.add('active');
}

function activateTabByPageNumber(pageNum) {
    const tabs = document.querySelectorAll('.pdf-tab');
    tabs.forEach((tab, index) => {
        if (index === pageNum - 1) {
            setActiveTab(tab);
        }
    });
}

function updateNavigationButtons(currentPage, numPages) {
    const previousButton = document.getElementById('previous-page-button');
    const nextButton = document.getElementById('next-page-button');

    if (numPages === 1 || numPages === undefined) { 
        // Si le document n'a qu'une seule page ou si le nombre de pages est indéfini (pour les images par exemple)
        previousButton.disabled = true;
        nextButton.disabled = true;
        previousButton.classList.add('disabled');
        nextButton.classList.add('disabled');
    } else {
        // Activer ou désactiver les boutons en fonction de la page actuelle
        previousButton.disabled = currentPage === 1;
        nextButton.disabled = currentPage === numPages;

        if (currentPage === 1) {
            previousButton.classList.add('disabled');
        } else {
            previousButton.classList.remove('disabled');
        }

        if (currentPage === numPages) {
            nextButton.classList.add('disabled');
        } else {
            nextButton.classList.remove('disabled');
        }
    }
}

// ajout v1.2 - gestion des sauvegardes //
function confirmBackup() {
    // Vérifiez que les traductions sont disponibles
    if (!translations || !translations.confirm_backup_message) {
        console.error("Erreur : la traduction pour 'confirm_backup_message' est manquante.");
        alert("Erreur : la traduction est manquante.");
        return;
    }

    // Afficher le message de confirmation avec la traduction
    const message = translations.confirm_backup_message;
    if (confirm(message)) {
        backupFiles(); // Appel à la fonction de sauvegarde
    }
}



async function backupFiles() {
    const infoForm = document.getElementById('infoForm');
    const loadingSpinner = document.getElementById('loadingSpinner');
    const backupProgress = document.querySelector('#backupModal .progress-bar');

    // Réinitialiser la barre de progression
    updateProgressBar(backupProgress, 0);

    // Désactiver le formulaire pendant la sauvegarde
    infoForm.querySelectorAll('input, textarea, button, select').forEach(element => {
        element.disabled = true;
    });

    // Afficher la modal et le spinner
    const backupModal = new bootstrap.Modal(document.getElementById('backupModal'));
    backupModal.show();
    loadingSpinner.style.display = 'block'; // Afficher le spinner
    backupProgress.style.display = 'none'; // Cacher la barre de progression au début

    try {
        // Démarrer la sauvegarde côté backend
        await window.electronAPI.backupFiles();
        showAlertMessage(translations["backup_success"], 'success'); // Utilisation des traductions
    } catch (error) {
        console.error("Erreur durant la sauvegarde:", error);
        showAlertMessage(translations["backup_failed"], 'failed'); // Utilisation des traductions
    } finally {
        // Réactiver le formulaire après la sauvegarde
        infoForm.querySelectorAll('input, textarea, button, select').forEach(element => {
            element.disabled = false;
        });

        setTimeout(() => {
            // Forcer le focus sur la fenêtre principale avant de fermer la modal
            window.focus();
            document.activeElement.blur();

            backupModal.hide();

            // Si la modal est toujours visible, utiliser dispose() pour forcer la fermeture
            const backupModalElement = document.getElementById('backupModal');
            if (backupModalElement.classList.contains('show')) {
                console.log('Forçage de la fermeture de la modal avec dispose()');
                bootstrap.Modal.getInstance(backupModalElement).dispose();
            }
        }, 4000); // Attendre un peu après la fin du backup avant de cacher la modal
    }
}

// Fonction de mise à jour de la barre de progression
function updateProgressBar(progressBar, progress) {
    if (progress === 0) {
        document.getElementById('loadingSpinner').style.display = 'block';  // Montrer le spinner au début
        progressBar.style.display = 'none'; // Cacher la barre de progression
    } else {
        document.getElementById('loadingSpinner').style.display = 'none';   // Cacher le spinner
        progressBar.style.display = 'block';   // Montrer la barre de progression
        progressBar.style.width = progress + '%';
        progressBar.setAttribute('aria-valuenow', progress);
    }
}

// Écouter les mises à jour de progression de la sauvegarde
window.electronAPI.on('backup-progress', (event, progress) => {
    updateProgressBar(document.querySelector('#backupModal .progress-bar'), progress);
});

// fermeture de la fenêtre d'état à la fin de la sauvegarde
window.electronAPI.on('backup-complete', () => {
    console.log('Backup terminé, fermeture de la fenêtre de sauvegarde...');
    
    setTimeout(() => {
        const backupModalElement = document.getElementById('backupModal');
        const backupModal = bootstrap.Modal.getInstance(backupModalElement);
        
        if (backupModal) {
            // Force le focus sur la fenêtre principale et ferme la modal
            window.focus();
            document.activeElement.blur();
            backupModal.hide();

            // Force la fermeture si la modal reste ouverte
            backupModal.dispose();
        }
    }, 500); // Attend un peu après la fin du backup pour permettre les animations
});

// Fonction pour afficher un message d'alerte personnalisé
function showAlertMessage(message, type) {
    const alertMessageElement = document.getElementById('alertMessage');
    alertMessageElement.textContent = message;

    // Supprime les classes d'alerte existantes
    alertMessageElement.classList.remove('alert-success', 'alert-failed', 'd-none');
    alertMessageElement.classList.add(`alert-${type}`);

    // Ajoute la classe d'alerte appropriée
    if (type === 'success') {
        alertMessageElement.classList.add('alert-success');
    } else if (type === 'failed') {
        alertMessageElement.classList.add('alert-failed');
    }

    // Affiche l'alerte
    setTimeout(() => {
        alertMessageElement.classList.add('d-none');
    }, 2000);
}

document.addEventListener('DOMContentLoaded', () => {
    window.electronAPI.onLogMessage((message) => {
        const previewPane = document.getElementById('preview');

        // Crée un nouvel élément div pour le message
        const messageElement = document.createElement('div');
        messageElement.textContent = message;

        // Ajouter le message à la prévisualisation
        previewPane.appendChild(messageElement);
    });
});

// EOF